
  /*
   *  Object %name    : %
   *  State           :  %state%
   *  Creation date   :  Sun Jul 22 10:17:10 2007
   *  Last modified   :  %modify_time%
   */
  /** @file 
   *  \brief A brief description of this module 
   *
   *  \version KMNG_ATP_TST_ManageOp.c#1:csrc:1
   *  \author NogaD
   *  \remarks Copyright (C) 2007 by Discretix Technologies Ltd.
   *           All Rights reserved
   */

/************* Include Files ****************/
#include "DX_VOS_BaseTypes.h"
#include "DX_VOS_Mem.h"
#include "CRYS.h"
#include "CRYS_KMNG.h"
#include "CRYS_AES.h"
#include "CRYS_DES.h"
#include "CRYS_error.h"
#include "CRYS_HASH.h"
#include "CRYS_HMAC.h"
#include "CRYS_RSA_Types.h"
#include "KMNG_Defs.h"
#include "KMNG_API.h"
#include "KMNG_ATP_TST_Defines.h"
#include "KMNG_TST_Utils.h"
#include "KMNG_ATP_TST_KeysOp.h"



/************************ Defines *******************************/
/************************ Enums *********************************/
/************************ Typedefs ******************************/
/************************ Global Data ***************************/

//DxUint32_t TST_KeyOperations_Buf[KMNG_TST_MAX_SIZE_KEY_RING_BUF];
KMNG_AES_WrappedKey_t   TST_AESWrappedKey;
KMNG_DES_WrappedKey_t   TST_DESWrappedKey;
KMNG_HMAC_WrappedKey_t  TST_HMACWrappedKey;
KMNG_RSA_WrappedKey_t   TST_RSAWrappedKey;
KMNG_DH_WrappedKey_t	  TST_DHWrappedKey;

extern DxUint32_t TST_KeyRing_Buf[KMNG_TST_MAX_SIZE_KEY_RING_BUF];

extern const DxUint32_t TST_NumOf_SymKeysOpTests;
extern const DxUint32_t TST_NumOf_RSAKeysOpTests;
extern const DxUint32_t TST_NumOf_DHKeysTests;

extern TST_KeysSym_TestDataStruct TST_KeysOpSym_TestData[];
extern TST_KeysRSA_TestDataStruct TST_KeysOpRSA_TestData[];
extern TST_KeysRSAHASHData_TestDataStruct TST_KeysOpRSAHASHData_TestData[];
extern TST_KeysDHData_TestDataStruct TST_KeysDHData_TestData[];

extern CRYS_DHUserPubKey_t TST_DHUserPubKey;
extern CRYS_DHPrimeData_t  TST_DHPrimeData;
TST_KeyDHServerData_Struct TST_KeyDHServerData;

/************* Private function prototype ***********************/
DxError_t KMNG_TST_Key_Ring_Initialization(DxUint32_t NumOfSymKeys, DxUint32_t NumOfRSAKeys, 
                                           DxUint32_t NumOfDHKeys, DxUint8_t* KeyRingPwd, 
                                           DxUint32_t PwdLen, DxUint8_t* KeyRing_ptr, 
                                           DxUint32_t KeyRingBufferSize);
/************************ Public Functions ******************************/

/*****************************************************************************
 * Function Name:                                                           
 *  KMNG_ATP_TST_KeysOperation      
 *  
 * Inputs:
 *  None
 *
 * Outputs: 
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *  This function Is an acceptance test for Key management API's. The function 
 *  tests the operations on keys.
 *  
 * Algorithm:       
 * 1. Initialize a keys ring
 * 2. Create and import symmetric and RSA keys
 * 3. Activate and use each of the keys
 * 4. Export and verify each of the keys
 * 
 *  * Implements KMNG ATP section 5.2 
 *
 ******************************************************************************/                                
DxError_t KMNG_ATP_TST_KeysOperation( void ) 
{

    /********* TST Variables Definitions ************/
    DxUint32_t            TST_index = 0;    
    DxError_t             TST_Status = DX_SUCCESS;        
    DxChar                TST_MasterTestName[TST_FUNC_STRING_LEN] = "KMNG_ATP_TST_Operations_2_Key_Ring";     
                              
    ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"********************************************************************* \n"));
    ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"************** KMNG_ATP_TST_Operations_2_Key_Ring Test ************** \n"));      
    ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"********************************************************************* \n \n"));
        
    /************ Initialize Key Ring ************/
    TST_Status = KMNG_TST_Key_Ring_Initialization(TST_NumOf_SymKeysOpTests, TST_NumOf_RSAKeysOpTests, 
                                                  TST_NumOf_RSAKeysOpTests, TST_KeysOpSym_TestData[0].TST_Pwd, 
                                                  TST_KeysOpSym_TestData[0].TST_PwdLen,
                                                  (DxUint8_t*)TST_KeyRing_Buf, KMNG_TST_MAX_SIZE_KEY_RING_BUF_BYTES);
                                           
    if (TST_Status != DX_SUCCESS)
        return KMNG_TEST_FAIL;     
    
    /************ Import and create Symmetric and RSA keys ************/
    for (TST_index = 0; TST_index < TST_NumOf_SymKeysOpTests; TST_index++)
    {
	  /* Import symmetric key (AES, DES & HMAC) */	  
      TST_Status = KMNG_ImportSymUserKey(TST_KeysOpSym_TestData[TST_index].TST_Pwd,
                                         TST_KeysOpSym_TestData[TST_index].TST_PwdLen,
                                         TST_KeysOpSym_TestData[TST_index].TST_KeyPwd, 
                                         TST_KeysOpSym_TestData[TST_index].TST_KeyPwdLen, 
                                         TST_KeysOpSym_TestData[TST_index].TST_KeyType,
                                         TST_KeysOpSym_TestData[TST_index].TST_KeySize,
                                         TST_KeysOpSym_TestData[TST_index].TST_KeyUsage, 
                                         TST_KeysOpSym_TestData[TST_index].TST_KeyRestriction,
                                         TST_KeysOpSym_TestData[TST_index].TST_UserData,
                                         TST_KeysOpSym_TestData[TST_index].TST_key,
                                         &TST_KeysOpSym_TestData[TST_index].TST_KeyID,
                                         (DxUint8_t*)TST_KeyRing_Buf); 
      
      TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                                "KMNG_ImportSymUserKey",
												         TST_KeysOpSym_TestData[TST_index].TST_Name,
												         TST_MasterTestName);
                                                 
      
      if (TST_index < TST_NumOf_RSAKeysOpTests)
      {                
    	/* Import RSA key */		
        TST_Status = KMNG_ImportRsaUserKey(TST_KeysOpRSA_TestData[TST_index].TST_Pwd, 
                                           TST_KeysOpRSA_TestData[TST_index].TST_PwdLen,
                                           TST_KeysOpRSA_TestData[TST_index].TST_KeyPwd ,
                                           TST_KeysOpRSA_TestData[TST_index].TST_KeyPwdLen,
                                           TST_KeysOpRSA_TestData[TST_index].TST_KeyType,
                                           TST_KeysOpRSA_TestData[TST_index].TST_KeySize,
                                           TST_KeysOpRSA_TestData[TST_index].TST_KeyUsage,
                                           TST_KeysOpRSA_TestData[TST_index].TST_KeyRestriction,
                                           TST_KeysOpRSA_TestData[TST_index].TST_UserData,
                                           TST_KeysOpRSA_TestData[TST_index].TST_N,
                                           TST_KeysOpRSA_TestData[TST_index].TST_NSize,
                                           TST_KeysOpRSA_TestData[TST_index].TST_E,
                                           TST_KeysOpRSA_TestData[TST_index].TST_ESize,
                                           TST_KeysOpRSA_TestData[TST_index].TST_D,
                                           TST_KeysOpRSA_TestData[TST_index].TST_DSize,
                                           &TST_KeysOpRSA_TestData[TST_index].TST_KeyID,
                                           (DxUint8_t*)TST_KeyRing_Buf); 
                                           
        TST_ATP_CHECK_ERROR_RETURN(TST_Status,
													        "KMNG_ImportRsaUserKey",
														      TST_KeysOpRSA_TestData[TST_index].TST_Name,
														      TST_MasterTestName);  
      }
      
      
      
      TST_index++;	  

	  /* Create symmetric key (AES, DES & HMAC) */
      TST_Status = KMNG_CreateUserKey(TST_KeysOpSym_TestData[TST_index].TST_Pwd , 
                                      TST_KeysOpSym_TestData[TST_index].TST_PwdLen,
                                      TST_KeysOpSym_TestData[TST_index].TST_KeyPwd, 
                                      TST_KeysOpSym_TestData[TST_index].TST_KeyPwdLen,
                                      TST_KeysOpSym_TestData[TST_index].TST_KeyType,
                                      TST_KeysOpSym_TestData[TST_index].TST_KeySize,
                                      TST_KeysOpSym_TestData[TST_index].TST_KeyUsage,
                                      TST_KeysOpSym_TestData[TST_index].TST_KeyRestriction,
                                      TST_KeysOpSym_TestData[TST_index].TST_UserData,
                                      &TST_KeysOpSym_TestData[TST_index].TST_KeyID,
                                      (DxUint8_t*)TST_KeyRing_Buf);
     
      TST_ATP_CHECK_ERROR_RETURN(TST_Status,
													      "KMNG_CreateUserKey",
													      TST_KeysOpSym_TestData[TST_index].TST_Name,
													      TST_MasterTestName);
                                                    
      if (TST_index < TST_NumOf_RSAKeysOpTests)
      {		
		/* Create RSA key */		
        TST_Status = KMNG_CreateUserKey(TST_KeysOpRSA_TestData[TST_index].TST_Pwd , 
                                        TST_KeysOpRSA_TestData[TST_index].TST_PwdLen,
                                        TST_KeysOpRSA_TestData[TST_index].TST_KeyPwd, 
                                        TST_KeysOpRSA_TestData[TST_index].TST_KeyPwdLen,
                                        TST_KeysOpRSA_TestData[TST_index].TST_KeyType,
                                        TST_KeysOpRSA_TestData[TST_index].TST_KeySize,
                                        TST_KeysOpRSA_TestData[TST_index].TST_KeyUsage,
                                        TST_KeysOpRSA_TestData[TST_index].TST_KeyRestriction,
                                        TST_KeysOpRSA_TestData[TST_index].TST_UserData,
                                        &TST_KeysOpRSA_TestData[TST_index].TST_KeyID,
                                        (DxUint8_t*)TST_KeyRing_Buf);
       
       TST_ATP_CHECK_ERROR_RETURN(TST_Status,
													       "KMNG_CreateUserKey",
													       TST_KeysOpRSA_TestData[TST_index].TST_Name,
													       TST_MasterTestName);
                                                   

      }
    }
    
    /************ Activate and Use Symmetric keys ************/   
    for (TST_index = 0; TST_index < TST_NumOf_SymKeysOpTests; TST_index++)
    {
      switch (TST_KeysOpSym_TestData[TST_index].TST_KeyType)
      {        
        case KMNG_KeyTypeAES:
          if(TST_KeysOpSym_TestData[TST_index].TST_KeyUsage != KMNG_KEY_USAGE_WRAPPING)
          {
            /* Activate and use AES key - non integrate functions */
            TST_Status = KMNG_TST_Activate_And_Use_AES_NonIntegrate(&TST_KeysOpSym_TestData[TST_index], 
                                                                        (DxUint8_t*)TST_KeyRing_Buf, 
                                                                        TST_KeysOpSym_TestData[TST_index].TST_VerifyOutput);
            if (TST_Status != DX_SUCCESS)
              return DX_FAILURE;
            /* Activate and use AES key - integrate functions */
            TST_Status = KMNG_TST_Activate_And_Use_AES_Integrate(&TST_KeysOpSym_TestData[TST_index], 
                                                                     (DxUint8_t*)TST_KeyRing_Buf, 
                                                                     TST_KeysOpSym_TestData[TST_index].TST_VerifyOutput);
            if (TST_Status != DX_SUCCESS)
              return DX_FAILURE;
          }
          else
          {
            TST_Status = KMNG_TST_Activate_And_Use_AES_WrapUnwrap(&TST_KeysOpSym_TestData[TST_index], 
                                                     (DxUint8_t*)TST_KeyRing_Buf, 
                                                     TST_KeysOpSym_TestData[TST_index].TST_VerifyOutput);
            if (TST_Status != DX_SUCCESS)
              return DX_FAILURE;
          }          
          break;
        case KMNG_KeyTypeDES:
          TST_Status = KMNG_TST_Activate_And_Use_DES_NonIntegrate(&TST_KeysOpSym_TestData[TST_index], 
                                                                      (DxUint8_t*)TST_KeyRing_Buf, 
                                                                      TST_KeysOpSym_TestData[TST_index].TST_VerifyOutput);
          if (TST_Status != DX_SUCCESS)
            return DX_FAILURE;
          /* Activate and use AES key - integrate functions */
          TST_Status = KMNG_TST_Activate_And_Use_DES_Integrate(&TST_KeysOpSym_TestData[TST_index], 
                                                                   (DxUint8_t*)TST_KeyRing_Buf, 
                                                                   TST_KeysOpSym_TestData[TST_index].TST_VerifyOutput);
          if (TST_Status != DX_SUCCESS)
            return DX_FAILURE;
          break;
        case KMNG_KeyTypeHMAC:
          /* Activate and use HMAC key - non integrate functions */
          #ifdef LLF_HASH_SHA384_SHA512_NOT_SUPPORTED
          if(TST_KeysOpSym_TestData[TST_index].TST_HASHMode == CRYS_HASH_SHA384_mode || TST_KeysOpSym_TestData[TST_index].TST_HASHMode == CRYS_HASH_SHA512_mode)
          {
            continue;
          }
          #endif
          
          TST_Status = KMNG_TST_Activate_And_Use_HMAC_NonIntegrate(&TST_KeysOpSym_TestData[TST_index], 
                                                                       (DxUint8_t*)TST_KeyRing_Buf, 
                                                                       TST_KeysOpSym_TestData[TST_index].TST_VerifyOutput);
          if (TST_Status != DX_SUCCESS)
            return DX_FAILURE;
          
          /* Activate and use HMAC key - integrate functions */
          TST_Status = KMNG_TST_Activate_And_Use_HMAC_Integrate(&TST_KeysOpSym_TestData[TST_index], 
                                                                    (DxUint8_t*)TST_KeyRing_Buf, 
                                                                    TST_KeysOpSym_TestData[TST_index].TST_VerifyOutput);
          if (TST_Status != DX_SUCCESS)
            return DX_FAILURE;
          
          break;
        
        default:
        
            return DX_FAILURE;
            }      
    }
    
    /************ Activate and Use RSA keys ************/   
    for (TST_index = 0; TST_index < TST_NumOf_RSAKeysOpTests; TST_index++)
    {
      #ifdef LLF_HASH_SHA384_SHA512_NOT_SUPPORTED
      if(TST_KeysOpRSA_TestData[TST_index].TST_HASHMode == CRYS_RSA_HASH_SHA384_mode || TST_KeysOpSym_TestData[TST_index].TST_HASHMode == CRYS_RSA_HASH_SHA512_mode)
      {
        continue;
      }
      #endif
      
      /* Activate and use RSA key - encrypt/decrypt */
      TST_Status = KMNG_TST_Activate_And_Use_RSA_EncryptDecrypt(&TST_KeysOpRSA_TestData[TST_index], 
                                                     (DxUint8_t*)TST_KeyRing_Buf, 
                                                     TST_KeysOpRSA_TestData[TST_index].TST_VerifyOutput);
      if (TST_Status != DX_SUCCESS)
        return DX_FAILURE;    
      
      /* Activate and use RSA key - sign verify non integrated functions */
      TST_Status = KMNG_TST_Activate_And_Use_RSA_SignVerify_NonIntegrated(&TST_KeysOpRSA_TestData[TST_index], 
                                                                              (DxUint8_t*)TST_KeyRing_Buf);
      if (TST_Status != DX_SUCCESS)
        return DX_FAILURE;
      
      /* Activate and use RSA key - sign verify integrated functions */
      TST_Status = KMNG_TST_Activate_And_Use_RSA_SignVerify(&TST_KeysOpRSA_TestData[TST_index], 
                                                                (DxUint8_t*)TST_KeyRing_Buf,
                                                                &TST_KeysOpRSAHASHData_TestData[TST_index]);
      if (TST_Status != DX_SUCCESS)
        return DX_FAILURE;
      
    }
        
    /************ Export keys and verify ************/       
    /* Export and verify symmetric keys */
    for (TST_index = 0; TST_index < TST_NumOf_SymKeysOpTests; TST_index++ )
    {      
	  /* Export symmetric key and verify the key */
      TST_Status = KMNG_TST_Export_and_Verify_SymKeys(&TST_KeysOpSym_TestData[TST_index], 
                                                          (DxUint8_t*)TST_KeyRing_Buf);
      if (TST_Status != DX_SUCCESS)
        return DX_FAILURE;                                                                                        
    }
    
    /* Export and verify RSA keys */
    for (TST_index = 0; TST_index < TST_NumOf_RSAKeysOpTests; TST_index++ )
    {      
	  /* Export symmetric key and verify the key */
      TST_Status = KMNG_TST_Export_and_Verify_RSAKeys(&TST_KeysOpRSA_TestData[TST_index], 
                                                          (DxUint8_t*)TST_KeyRing_Buf);
      if (TST_Status != DX_SUCCESS)
        return DX_FAILURE;                                                                                        
    }
    
    return DX_SUCCESS;

EXIT_ON_ERROR:
    {
      return DX_FAILURE;
    }
}

/***************************************************************************************************************************************************************/
/***TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************************************************/
/***--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************************************************/
/***TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************************************************/
/***--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************************************************/
/***************************************************************************************************************************************************************/


/*****************************************************************************
* Function Name:                                                           
*  KMNG_ATP_TST_KeysOperation      
*  
* Inputs:
*  None
*
* Outputs: 
*  CRYSError_t - Function error return                                     
*
* Description:                                                                 
*  This function Is an acceptance test for Key management API's. The function 
*  tests the operations on keys.
*  
* Algorithm:       
* 1. Initialize a keys ring
* 2. Import and create DH keys
* 3. Export the created DH key
* 4. Create PUB & Priv key in the CRYS (the Server Public key)
* 5. Activate the KMNG key and get the secret key
* 6. Get the server secret key (using the CRYS)
* 7. Compare the 2 secret keys - should be equal
* 
*  * Implements KMNG ATP section 5.3 
*
******************************************************************************/                                
DxError_t KMNG_ATP_TST_DH_KeysOperation( void ) 
{

	/********* TST Variables Definitions ************/
	DxUint32_t            TST_index = 0;    
	DxError_t             TST_Status = DX_SUCCESS;        
	DxChar                TST_MasterTestName[TST_FUNC_STRING_LEN] = "KMNG_ATP_TST_DH_KeysOperation";     
  DxUint16_t            TST_ServerSecretKeySize = TST_KMNG_MAX_DH_PARAM_SIZE;


	ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"********************************************************************* \n"));
	ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"************** KMNG_ATP_TST_DH_KeysOperation Test ************** \n"));      
	ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"********************************************************************* \n \n"));

	/************ Initialize Key Ring ************/
	TST_Status = KMNG_TST_Key_Ring_Initialization(TST_NumOf_SymKeysOpTests, TST_NumOf_RSAKeysOpTests, 
		                                            TST_NumOf_DHKeysTests, TST_KeysOpSym_TestData[0].TST_Pwd, 
		                                            TST_KeysOpSym_TestData[0].TST_PwdLen,
		                                            (DxUint8_t*)TST_KeyRing_Buf, KMNG_TST_MAX_SIZE_KEY_RING_BUF_BYTES);

	if (TST_Status != DX_SUCCESS)
		return DX_FAILURE;     

	/**** Main Loop ****/
	for (TST_index = 0; TST_index < TST_NumOf_DHKeysTests; TST_index++)
	{		
		    TST_ServerSecretKeySize = TST_KMNG_MAX_DH_PARAM_SIZE;  
        /************ Import and create DH keys ************/
        if (TST_index % 2 == 0)
        {
            /* Create DH key */
		        TST_Status = KMNG_CreateUserKey(TST_KeysDHData_TestData[TST_index].TST_Pwd , 
	                                          TST_KeysDHData_TestData[TST_index].TST_PwdLen,
	                                          TST_KeysDHData_TestData[TST_index].TST_KeyPwd, 
	                                          TST_KeysDHData_TestData[TST_index].TST_KeyPwdLen,
	                                          TST_KeysDHData_TestData[TST_index].TST_KeyType,
	                                          TST_KeysDHData_TestData[TST_index].TST_KeySize,
	                                          TST_KeysDHData_TestData[TST_index].TST_KeyUsage,
	                                          TST_KeysDHData_TestData[TST_index].TST_KeyRestriction,
	                                          TST_KeysDHData_TestData[TST_index].TST_UserData,
	                                          &TST_KeysDHData_TestData[TST_index].TST_KeyID,
	                                          (DxUint8_t*)TST_KeyRing_Buf);

		        TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                                      "KMNG_CreateUserKey",
                                      TST_KeysDHData_TestData[TST_index].TST_Name,
                                      TST_MasterTestName);
        }
        else
        {
            /* for each loop do either create or import key */                    
            TST_Status = KMNG_ImportDHUserKey(TST_KeysDHData_TestData[TST_index].TST_Pwd,
                                              TST_KeysDHData_TestData[TST_index].TST_PwdLen,
                                              TST_KeysDHData_TestData[TST_index].TST_KeyPwd, 
                                              TST_KeysDHData_TestData[TST_index].TST_KeyPwdLen,
                                              TST_KeysDHData_TestData[TST_index].TST_KeyType,
                                              TST_KeysDHData_TestData[TST_index].TST_KeySize,
                                              TST_KeysDHData_TestData[TST_index].TST_KeyUsage,
                                              TST_KeysDHData_TestData[TST_index].TST_KeyRestriction,
                                              TST_KeysDHData_TestData[TST_index].TST_UserData,
                                              TST_KeysDHData_TestData[TST_index].TST_P,
                                              TST_KeysDHData_TestData[TST_index].TST_PSize,
                                              TST_KeysDHData_TestData[TST_index].TST_G,
                                              TST_KeysDHData_TestData[TST_index].TST_GSize,
                                              TST_KeysDHData_TestData[TST_index].TST_PubKey,
                                              TST_KeysDHData_TestData[TST_index].TST_PubKeySize,
                                              TST_KeysDHData_TestData[TST_index].TST_PrivateKey,
                                              TST_KeysDHData_TestData[TST_index].TST_PrivateKeySize,
                                              &TST_KeysDHData_TestData[TST_index].TST_KeyID,
                                              (DxUint8_t*)TST_KeyRing_Buf);
            TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                                      "KMNG_ImportDHUserKey",
                                      TST_KeysDHData_TestData[TST_index].TST_Name,
                                      TST_MasterTestName);
        }
        
        /************ Export the created DH key ************/
        TST_Status = KMNG_TST_Export_and_Verify_DHKey(&TST_KeysDHData_TestData[TST_index],(DxUint8_t*)TST_KeyRing_Buf);
       
        if (TST_Status != DX_SUCCESS)
            return DX_FAILURE;
        
        /****** Create PUB & Priv key in the CRYS - Server Public key ******/        
        TST_KeyDHServerData.TST_ServerPrivKeySize = TST_KMNG_MAX_DH_PARAM_SIZE;
        TST_KeyDHServerData.TST_ServerPubKeySize = TST_KMNG_MAX_DH_PARAM_SIZE;
        
        TST_Status =  CRYS_DH_PKCS3_GeneratePubPrv(TST_KeysDHData_TestData[TST_index].TST_ExportedG,
                                                  (DxUint16_t)TST_KeysDHData_TestData[TST_index].TST_ExportedGSize,
                                                  TST_KeysDHData_TestData[TST_index].TST_ExportedP,
                                                  (DxUint16_t)TST_KeysDHData_TestData[TST_index].TST_ExportedPSize,
                                                  TST_KeysDHData_TestData[TST_index].TST_L,
                                                  &TST_DHUserPubKey,
                                                  &TST_DHPrimeData,
                                                  TST_KeyDHServerData.TST_ServerPrivKey,
                                                  &TST_KeyDHServerData.TST_ServerPrivKeySize,
                                                  TST_KeyDHServerData.TST_ServerPubKey,
                                                  &TST_KeyDHServerData.TST_ServerPubKeySize);

        TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                                  "CRYS_DH_PKCS3_GeneratePubPrv",
                                  TST_KeysDHData_TestData[TST_index].TST_Name,
                                  TST_MasterTestName);
        
        /****** Activate the KMNG key and get the secret key *******/
        TST_Status = KMNG_TST_Activate_And_Use_DH_GetSecretKey(&TST_KeysDHData_TestData[TST_index], 
                                                               (DxUint8_t*)TST_KeyRing_Buf, &TST_KeyDHServerData);

        if (TST_Status != DX_SUCCESS)
            return DX_FAILURE;       
        
        /****** Get the server secret key (using the CRYS) *******/
        TST_Status = CRYS_DH_PKCS3_GetSecretKey(TST_KeyDHServerData.TST_ServerPrivKey,
                                                TST_KeyDHServerData.TST_ServerPrivKeySize,
                                                TST_KeysDHData_TestData[TST_index].TST_ExportedPubKey,
                                                (DxUint16_t)TST_KeysDHData_TestData[TST_index].TST_ExportedPubKeySize,
                                                TST_KeysDHData_TestData[TST_index].TST_ExportedP,
                                                (DxUint16_t)TST_KeysDHData_TestData[TST_index].TST_ExportedPSize,
                                                &TST_DHUserPubKey, &TST_DHPrimeData,
                                                TST_KeysDHData_TestData[TST_index].TST_ServerSecretKey,
                                                &TST_ServerSecretKeySize);

        TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                                  "CRYS_DH_PKCS3_GetSecretKey",
                                  TST_KeysDHData_TestData[TST_index].TST_Name,
                                  TST_MasterTestName);



        /*Compare the 2 secret keys */
        TST_Status = DX_VOS_MemCmp(TST_KeysDHData_TestData[TST_index].TST_ServerSecretKey, TST_KeysDHData_TestData[TST_index].TST_SecretKey,
                                   TST_ServerSecretKeySize);

        if (TST_Status != DX_SUCCESS)
        {
            ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"!!!! TST_Error - Secret keys are different !!!\n"));
            return DX_FAILURE; 
        }
  }

	return DX_SUCCESS;

EXIT_ON_ERROR:
  {
    return DX_FAILURE;
  }
}

/***************************************************************************************************************************************************************/
/***TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************************************************/
/***--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************************************************/
/***TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************************************************/
/***--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************************************************/
/***************************************************************************************************************************************************************/


/*****************************************************************************
* Function Name:                                                           
*  KMNG_ATP_TST_KeysOperation      
*  
* Inputs:
*  None
*
* Outputs: 
*  CRYSError_t - Function error return                                     
*
* Description:                                                                 
*  This function Is an acceptance test for Key management API's. The function 
*  tests the operations on keys.
*  
* Algorithm:       
* 1. Initialize a key ring
* 2. Import DH keys
* 3. Get the Public key from the Key Ring
* 4. Verify that the returned public key is as expected
* 
*  * Implements KMNG ATP section 5.4 
*
******************************************************************************/                                
DxError_t KMNG_ATP_TST_DH_GetPubKey( void ) 
{

    /********* TST Variables Definitions ************/
    DxUint32_t            TST_index = 0;    
    DxError_t             TST_Status = DX_SUCCESS;        
    DxChar                TST_MasterTestName[TST_FUNC_STRING_LEN] = "KMNG_ATP_TST_DH_GetPubKey";         

    ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"********************************************************************* \n"));
    ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"************** KMNG_ATP_TST_DH_GetPubKey Test ************** \n"));      
    ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"********************************************************************* \n \n"));

    /************ Initialize Key Ring ************/
    TST_Status = KMNG_TST_Key_Ring_Initialization(TST_NumOf_SymKeysOpTests, TST_NumOf_RSAKeysOpTests, 
                                                  TST_NumOf_DHKeysTests, TST_KeysOpSym_TestData[0].TST_Pwd, 
                                                  TST_KeysOpSym_TestData[0].TST_PwdLen,
                                                  (DxUint8_t*)TST_KeyRing_Buf, KMNG_TST_MAX_SIZE_KEY_RING_BUF_BYTES);

    if (TST_Status != DX_SUCCESS)
        return DX_FAILURE;     

    /**** Main Loop ****/
    for (TST_index = 0; TST_index < TST_NumOf_DHKeysTests; TST_index++)
    {		        
        /*Only test for Imported vectors */
        if (TST_index % 2 == 0)
            continue;
        /************ Import DH keys ************/                       
        TST_Status = KMNG_ImportDHUserKey(TST_KeysDHData_TestData[TST_index].TST_Pwd,
                                          TST_KeysDHData_TestData[TST_index].TST_PwdLen,
                                          TST_KeysDHData_TestData[TST_index].TST_KeyPwd, 
                                          TST_KeysDHData_TestData[TST_index].TST_KeyPwdLen,
                                          TST_KeysDHData_TestData[TST_index].TST_KeyType,
                                          TST_KeysDHData_TestData[TST_index].TST_KeySize,
                                          TST_KeysDHData_TestData[TST_index].TST_KeyUsage,
                                          TST_KeysDHData_TestData[TST_index].TST_KeyRestriction,
                                          TST_KeysDHData_TestData[TST_index].TST_UserData,
                                          TST_KeysDHData_TestData[TST_index].TST_P,
                                          TST_KeysDHData_TestData[TST_index].TST_PSize,
                                          TST_KeysDHData_TestData[TST_index].TST_G,
                                          TST_KeysDHData_TestData[TST_index].TST_GSize,
                                          TST_KeysDHData_TestData[TST_index].TST_PubKey,
                                          TST_KeysDHData_TestData[TST_index].TST_PubKeySize,
                                          TST_KeysDHData_TestData[TST_index].TST_PrivateKey,
                                          TST_KeysDHData_TestData[TST_index].TST_PrivateKeySize,
                                          &TST_KeysDHData_TestData[TST_index].TST_KeyID,
                                          (DxUint8_t*)TST_KeyRing_Buf);
        TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                                   "KMNG_ImportDHUserKey",
                                   TST_KeysDHData_TestData[TST_index].TST_Name,
                                   TST_MasterTestName);

    
        /************ Get the Public key from the Key Ring ************/
        TST_KeysDHData_TestData[TST_index].TST_ExportedPSize = TST_KMNG_MAX_DH_PARAM_SIZE;
        TST_KeysDHData_TestData[TST_index].TST_ExportedGSize = TST_KMNG_MAX_DH_PARAM_SIZE;
        TST_KeysDHData_TestData[TST_index].TST_ExportedPubKeySize = TST_KMNG_MAX_DH_PARAM_SIZE;

        TST_Status = KMNG_GetDHPublicKey(TST_KeysDHData_TestData[TST_index].TST_Pwd,
                                         TST_KeysDHData_TestData[TST_index].TST_PwdLen,
                                         TST_KeysDHData_TestData[TST_index].TST_KeyPwd, 
                                         TST_KeysDHData_TestData[TST_index].TST_KeyPwdLen,
                                         TST_KeysDHData_TestData[TST_index].TST_KeyID,
                                         TST_KeysDHData_TestData[TST_index].TST_ExportedP,
                                         &TST_KeysDHData_TestData[TST_index].TST_ExportedPSize,
                                         TST_KeysDHData_TestData[TST_index].TST_ExportedG,
                                         &TST_KeysDHData_TestData[TST_index].TST_ExportedGSize,
                                         TST_KeysDHData_TestData[TST_index].TST_ExportedPubKey,
                                         &TST_KeysDHData_TestData[TST_index].TST_ExportedPubKeySize,
                                         (DxUint8_t*)TST_KeyRing_Buf);

        TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                                  "KMNG_GetDHPublicKey",
                                  TST_KeysDHData_TestData[TST_index].TST_Name,
                                  TST_MasterTestName);

        /************ Verify that the returned public key is as expected ************/        
        /* Verify P & P size */
        if (TST_KeysDHData_TestData[TST_index].TST_PSize != TST_KeysDHData_TestData[TST_index].TST_ExportedPSize)        
            goto EXIT_WITH_ERROR;        

        TST_Status = (DxUint32_t)DX_VOS_MemCmp((DxUint8_t*)TST_KeysDHData_TestData[TST_index].TST_ExportedP,
                                               (DxUint8_t*)TST_KeysDHData_TestData[TST_index].TST_P, 
                                                TST_KeysDHData_TestData[TST_index].TST_PSize);
        if (TST_Status != DX_SUCCESS)
            goto EXIT_WITH_ERROR;

        /* Verify G & G size */
        if (TST_KeysDHData_TestData[TST_index].TST_GSize != TST_KeysDHData_TestData[TST_index].TST_ExportedGSize)        
            goto EXIT_WITH_ERROR;        

        TST_Status = (DxUint32_t)DX_VOS_MemCmp((DxUint8_t*)TST_KeysDHData_TestData[TST_index].TST_ExportedG,
                                               (DxUint8_t*)TST_KeysDHData_TestData[TST_index].TST_G, 
                                                TST_KeysDHData_TestData[TST_index].TST_GSize);
        if (TST_Status != DX_SUCCESS)
            goto EXIT_WITH_ERROR;

        /* Verify PubKey & pubkey size */
        if (TST_KeysDHData_TestData[TST_index].TST_ExportedPubKeySize != TST_KeysDHData_TestData[TST_index].TST_PubKeySize)        
            goto EXIT_WITH_ERROR;        

        TST_Status = (DxUint32_t)DX_VOS_MemCmp((DxUint8_t*)TST_KeysDHData_TestData[TST_index].TST_ExportedPubKey,
                                               (DxUint8_t*)TST_KeysDHData_TestData[TST_index].TST_PubKey, 
                                               TST_KeysDHData_TestData[TST_index].TST_PubKeySize);
        if (TST_Status != DX_SUCCESS)
            goto EXIT_WITH_ERROR;
    }
    return DX_SUCCESS;

EXIT_WITH_ERROR:
    ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"Exported value is different than expected \n"));
    return KMNG_TEST_FAIL;

EXIT_ON_ERROR:
    {
      return DX_FAILURE;
    }
}

/***************************************************************************************************************************************************************/
/***TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************************************************/
/***--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************************************************/
/***TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************************************************/
/***--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************************************************/
/***************************************************************************************************************************************************************/




/*****************************************************************************
 * Function Name:KMNG_TST_Export_and_Verify_SymKeys
 *  
 * Inputs:
 *  None
 *
 * Outputs: 
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *  This function activate a key from a given key ring and uses it.
 *  
 * Algorithm:       
 * 1. Export the symmetric key
 * 2. Verify that the key is as expected
 *
 ******************************************************************************/
DxError_t KMNG_TST_Export_and_Verify_SymKeys(TST_KeysSym_TestDataStruct* KeyData, 
                                             DxUint8_t* KeyRing)
{
  /********* TST Variables Definitions ************/
  DxError_t             TST_Status = DX_SUCCESS;        
  DxChar                TST_MasterTestName[TST_FUNC_STRING_LEN] = "KMNG_TST_Export_and_Verify_SymKeys"; 
  KMNG_KeyType_t        TST_KeyType;
  DxUint32_t            TST_KeyUsage;
  DxUint32_t            TST_KeyRestriction;
  DxUint32_t            TST_KeySize = 0;  
  
  /********* Export symmetric key ************/     
  TST_Status = KMNG_ExportSymUserKey(KeyData->TST_Pwd , 
                                     KeyData->TST_PwdLen,
                                     KeyData->TST_KeyPwd, 
                                     KeyData->TST_KeyPwdLen,
                                     KeyData->TST_KeyID,
                                     KeyData->TST_Exportedkey,
                                     &TST_KeyType,
                                     &TST_KeySize,
                                     &TST_KeyUsage,
                                     &TST_KeyRestriction,
                                     KeyRing);
      
  TST_ATP_CHECK_ERROR_RETURN(TST_Status,
											      "KMNG_ExportSymUserKey",
											      KeyData->TST_Name,
											      TST_MasterTestName);
                                                   

    
  /********** Verify exported data **********/
  /* Verify key type */
  if (TST_KeyType != KeyData->TST_KeyType)
  {
    ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"Exported key is different than expected \n"));
    return (DxError_t)KMNG_TEST_FAIL;  
  }
  /* Verify key size */
  if (TST_KeySize != KeyData->TST_KeySize)
  {
    ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"Exported key is different than expected \n"));
    return (DxError_t)KMNG_TEST_FAIL;  
  }
  /* Verify key usage */
  if (TST_KeyUsage != KeyData->TST_KeyUsage)
  {
    ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"Exported key is different than expected \n"));
    return (DxError_t)KMNG_TEST_FAIL;  
  }
  /* Verify key restriction */
  if (TST_KeyRestriction != KeyData->TST_KeyRestriction)
  {
    ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"Exported key is different than expected \n"));
    return (DxError_t)KMNG_TEST_FAIL;  
  }

  /* Verify key data*/
  TST_Status = (DxUint32_t)DX_VOS_MemCmp((DxUint8_t*)KeyData->TST_Exportedkey,(DxUint8_t*)KeyData->TST_key, 
                                         TST_KeySize*sizeof(DxUint8_t));
  /* comparison should fail in case of created key */
  if (TST_Status != DX_SUCCESS)
  {
    if (KeyData->TST_VerifyOutput != TST_KMNG_DO_NOT_VERIFY_OUTPUT)
    {
      ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"Exported key is different than expected \n"));
      return (DxError_t)KMNG_TEST_FAIL;    
    }
  }
  else
  {
    if (KeyData->TST_VerifyOutput != TST_KMNG_DO_NOT_VERIFY_OUTPUT)
    {
      return DX_SUCCESS;    
    }  
  }
  
  return DX_SUCCESS; 

EXIT_ON_ERROR:
{
  return DX_FAILURE;

}
}

/*****************************************************************************
 * Function Name:KMNG_TST_Export_and_Verify_RSAKeys
 *  
 * Inputs:
 *  None
 *
 * Outputs: 
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *  This function activate a key from a given key ring and uses it.
 *  
 * Algorithm:       
 * 1. Export the RSA key
 * 2. Verify that the key is as expected
 *
 ******************************************************************************/
DxError_t KMNG_TST_Export_and_Verify_RSAKeys(TST_KeysRSA_TestDataStruct* KeyData, 
                                             DxUint8_t* KeyRing)
{
  /********* TST Variables Definitions ************/
  DxError_t             TST_Status = DX_SUCCESS;        
  DxChar                TST_MasterTestName[TST_FUNC_STRING_LEN] = "KMNG_TST_Export_and_Verify_RSAKeys";  
  
  /********* Export symmetric key ************/     
  KeyData->TST_ExportedNSize = TST_KMNG_RSA_2048_KEYSIZE*TST_KMNG_BITS_IN_BYTE;
  KeyData->TST_ExportedDSize = TST_KMNG_RSA_2048_KEYSIZE*TST_KMNG_BITS_IN_BYTE;
  KeyData->TST_ExportedESize = TST_KMNG_BITS_IN_BYTE*4;

  TST_Status = KMNG_ExportRSAUserKey(KeyData->TST_Pwd , 
                                     KeyData->TST_PwdLen,
                                     KeyData->TST_KeyPwd, 
                                     KeyData->TST_KeyPwdLen,
                                     KeyData->TST_KeyID,
                                     KeyData->TST_ExportedN,
                                     &KeyData->TST_ExportedNSize,
                                     KeyData->TST_ExportedE,
                                     &KeyData->TST_ExportedESize,
                                     KeyData->TST_ExportedD,
                                     &KeyData->TST_ExportedDSize,                                    
                                     KeyRing);
      
  TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                            "KMNG_ExportRSAUserKey",
                            KeyData->TST_Name,
                            TST_MasterTestName);
                                                  
    
  /********** Verify exported data **********/  
  TST_Status = (DxUint32_t)DX_VOS_MemCmp((DxUint8_t*)KeyData->TST_ExportedE,(DxUint8_t*)KeyData->TST_E, 
                             KeyData->TST_ExportedESize*sizeof(DxUint8_t));
  /* comparison should fail in case of created key */
  if (TST_Status != DX_SUCCESS)
  {
    if (KeyData->TST_VerifyOutput != TST_KMNG_DO_NOT_VERIFY_OUTPUT)
    {
      ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"Exported E is different than expected \n"));
      return DX_FAILURE;    
    }
  }
 
  TST_Status = (DxUint32_t)DX_VOS_MemCmp((DxUint8_t*)KeyData->TST_ExportedN,(DxUint8_t*)KeyData->TST_N, 
                             KeyData->TST_ExportedNSize*sizeof(DxUint8_t));
  /* comparison should fail in case of created key */
  if (TST_Status != DX_SUCCESS)
  {
    if (KeyData->TST_VerifyOutput != TST_KMNG_DO_NOT_VERIFY_OUTPUT)
    {
      ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"Exported N is different than expected \n"));
      return DX_FAILURE;    
    }
  }
 
  TST_Status = (DxUint32_t)DX_VOS_MemCmp((DxUint8_t*)KeyData->TST_ExportedD,(DxUint8_t*)KeyData->TST_D, 
                             KeyData->TST_ExportedDSize*sizeof(DxUint8_t));
  /* comparison should fail in case of created key */
  if (TST_Status != DX_SUCCESS)
  {
    if (KeyData->TST_VerifyOutput != TST_KMNG_DO_NOT_VERIFY_OUTPUT)
    {
      ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"Exported D is different than expected \n"));
      return DX_FAILURE;    
    }
  }
  
  return DX_SUCCESS; 

EXIT_ON_ERROR:
  {
    return DX_FAILURE;
  }
}


/*****************************************************************************
* Function Name:KMNG_TST_Export_and_Verify_DHKey
*  
* Inputs:
*  None
*
* Outputs: 
*  CRYSError_t - Function error return                                     
*
* Description:                                                                 
*  This function activate a key from a given key ring and uses it.
*  
* Algorithm:       
* 1. Export the DH key
* 2. Verify that the key is as expected
*
******************************************************************************/
DxError_t KMNG_TST_Export_and_Verify_DHKey(TST_KeysDHData_TestDataStruct* KeyData, 
                                           DxUint8_t* KeyRing)
{
    /********* TST Variables Definitions ************/
    DxError_t             TST_Status = DX_SUCCESS;        
    DxChar                TST_MasterTestName[TST_FUNC_STRING_LEN] = "KMNG_TST_Export_and_Verify_SymKeys"; 

    /********* Export symmetric key ************/     
    
    KeyData->TST_ExportedGSize = TST_KMNG_MAX_DH_PARAM_SIZE;
    KeyData->TST_ExportedPSize = TST_KMNG_MAX_DH_PARAM_SIZE;
    KeyData->TST_ExportedPubKeySize = TST_KMNG_MAX_DH_PARAM_SIZE;
    KeyData->TST_ExportedPrivateKeySize = TST_KMNG_MAX_DH_PARAM_SIZE;

    TST_Status = KMNG_ExportDHUserKey(KeyData->TST_Pwd,
                                      KeyData->TST_PwdLen,
                                      KeyData->TST_KeyPwd, 
                                      KeyData->TST_KeyPwdLen,
                                      KeyData->TST_KeyID,
                                      KeyData->TST_ExportedP,
                                      &KeyData->TST_ExportedPSize,
                                      KeyData->TST_ExportedG,
                                      &KeyData->TST_ExportedGSize,
                                      KeyData->TST_ExportedPubKey,
                                      &KeyData->TST_ExportedPubKeySize,
                                      KeyData->TST_ExportedPrivateKey,
                                      &KeyData->TST_ExportedPrivateKeySize,
                                      KeyRing);

    TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                              "KMNG_ExportDHUserKey",
                              KeyData->TST_Name,
                              TST_MasterTestName);

    /********** Verify exported data **********/
    if (KeyData->TST_VerifyOutput == TST_KMNG_VERIFY_OUTPUT)
    {
        /* Verify P & P size */
        if (KeyData->TST_PSize != KeyData->TST_ExportedPSize)        
            goto EXIT_WITH_ERROR;        
        
        TST_Status = (DxUint32_t)DX_VOS_MemCmp((DxUint8_t*)KeyData->TST_ExportedP,(DxUint8_t*)KeyData->TST_P, 
                                               KeyData->TST_PSize);
        if (TST_Status != DX_SUCCESS)
            goto EXIT_WITH_ERROR;

        /* Verify G & G size */
        if (KeyData->TST_GSize != KeyData->TST_ExportedGSize)        
            goto EXIT_WITH_ERROR;        

        TST_Status = (DxUint32_t)DX_VOS_MemCmp((DxUint8_t*)KeyData->TST_ExportedG,(DxUint8_t*)KeyData->TST_G, 
            KeyData->TST_GSize);
        if (TST_Status != DX_SUCCESS)
            goto EXIT_WITH_ERROR;

        /* Verify PubKey & pubkey size */
        if (KeyData->TST_ExportedPubKeySize != KeyData->TST_PubKeySize)        
            goto EXIT_WITH_ERROR;        

        TST_Status = (DxUint32_t)DX_VOS_MemCmp((DxUint8_t*)KeyData->TST_ExportedPubKey,(DxUint8_t*)KeyData->TST_PubKey, 
            KeyData->TST_PubKeySize);
        if (TST_Status != DX_SUCCESS)
            goto EXIT_WITH_ERROR;

        /* Verify PrivateKey & PrivateKey size */
        if (KeyData->TST_ExportedPrivateKeySize != KeyData->TST_PrivateKeySize)        
            goto EXIT_WITH_ERROR;        

        TST_Status = (DxUint32_t)DX_VOS_MemCmp((DxUint8_t*)KeyData->TST_ExportedPrivateKey,
                                               (DxUint8_t*)KeyData->TST_PrivateKey, KeyData->TST_PrivateKeySize);
        if (TST_Status != DX_SUCCESS)
            goto EXIT_WITH_ERROR;
    }
    
    return DX_SUCCESS;         
    
EXIT_WITH_ERROR:
    ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"Exported data is different than expected \n"));
    return DX_FAILURE;

EXIT_ON_ERROR:
{
  return DX_FAILURE;
}
}

/*****************************************************************************
 * Function Name:                                                           
 *  KMNG_TST_Activate_And_Use_AES_NonIntegrate      
 *  
 * Inputs:
 *  None
 *
 * Outputs: 
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *  This function activate a key from a given key ring and uses it.
 *  
 * Algorithm:       
 * 1. Activate the AES key
 * 2. Initialize encryption operation
 * 3. Encrypt data block
 * 4. Compare the output to expected data (if required)
 * 5. Finish the operation
 * 6. Repeat steps 2-5 for decrypt operation  
 *
 ******************************************************************************/
DxError_t KMNG_TST_Activate_And_Use_AES_NonIntegrate(TST_KeysSym_TestDataStruct* KeyData, 
                                                     DxUint8_t* KeyRing, DxUint8_t VerifyOutput)
{
  /********* TST Variables Definitions ************/
  DxError_t               TST_Status = DX_SUCCESS;
  CRYS_AESUserContext_t   TST_AesContext;  
  DxChar                  TST_MasterTestName[TST_FUNC_STRING_LEN] = "KMNG_TST_Activate_And_Use_AES_NonIntegrate";  
  DxUint8_t               TST_Output[TST_KMNG_MAX_SIZE_DATA]; 
  DxUint8_t*              TST_Input_ptr;
  DxUint8_t               TST_index; 
  CRYS_AES_EncryptMode_t  TST_OpMode;
  
  ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"********************************************************************* \n"));
  ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"Activate and use %s \n",KeyData->TST_Name));      
  ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"********************************************************************* \n"));

  /********* Activate AES key  ************/  
  TST_Status = KMNG_ActivateAESKey(KeyData->TST_Pwd, KeyData->TST_PwdLen,
                                   KeyData->TST_KeyPwd, KeyData->TST_KeyPwdLen,
                                   KeyData->TST_KeyID, KeyRing,
                                   TST_AESWrappedKey);

  TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                             "KMNG_ActivateAESKey",
                             KeyData->TST_Name,
                             TST_MasterTestName);
                                             

  
  /********* Initialize the activated AES key ************/    
  /********* Encrypt\Decrypt operation ************/
  for(TST_index = 0; TST_index < CRYS_AES_EncryptNumOfOptions; TST_index++)
  {
    if (TST_index == 0)
      TST_OpMode = CRYS_AES_Encrypt;
    else
      TST_OpMode = CRYS_AES_Decrypt;
    
    /* In AES MAC mode there is no decryption only encryption */
    if (KeyData->TST_AESMode == CRYS_AES_MAC_mode && TST_OpMode == CRYS_AES_Decrypt)
      break;
    
    /* In AES CTR mode there is only encrypt */
    if (KeyData->TST_AESMode == CRYS_AES_CTR_mode && TST_OpMode == CRYS_AES_Decrypt)
      TST_OpMode = CRYS_AES_Encrypt;
    
    /* Initialize AES context */  
    TST_Status = CRYS_KMNG_AES_Init(TST_AESWrappedKey,
  										             &TST_AesContext,
  										             KeyData->TST_IV,
  										             TST_OpMode,
  										             KeyData->TST_AESMode);
  										              
    TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                               "CRYS_KMNG_AES_Init",
                               KeyData->TST_Name,
                               TST_MasterTestName); 
    
    /* copy input to output in case of encrypt*/
    if (TST_OpMode == CRYS_AES_Encrypt)
    {      
      DX_VOS_FastMemCpy(TST_Output, KeyData->TST_Data, KeyData->TST_DataLen); 
      TST_Input_ptr = KeyData->TST_Data; 
    }
    else
    {
      TST_Input_ptr = KeyData->TST_OutputData; 
    }
    
    
    /* Encrypt\Decrypt data */
    TST_Status = CRYS_AES_Block(&TST_AesContext,   
                               TST_Input_ptr,     
                               KeyData->TST_DataLen,
                               KeyData->TST_OutputData); 
    
    TST_ATP_CHECK_ERROR_RETURN((DxError_t)TST_Status,
                               "CRYS_AES_Block",
                               KeyData->TST_Name,
                               TST_MasterTestName);

    /* Verify the output if needed */
    if (VerifyOutput == TST_KMNG_VERIFY_OUTPUT || TST_OpMode == CRYS_AES_Decrypt)
    {
      if (TST_OpMode == CRYS_AES_Encrypt)
      {
        if (KeyData->TST_AESMode == CRYS_AES_MAC_mode)
          TST_Status = (DxUint32_t)DX_VOS_MemCmp(KeyData->TST_ExpectedOutput,KeyData->TST_OutputData,TST_KMNG_MAC_SIZE_BYTE);
        else
          TST_Status = (DxUint32_t)DX_VOS_MemCmp(KeyData->TST_ExpectedOutput,KeyData->TST_OutputData,KeyData->TST_DataLen);
      }
      else
        TST_Status = (DxUint32_t)DX_VOS_MemCmp(KeyData->TST_Data,KeyData->TST_OutputData,KeyData->TST_DataLen);  
      
      if (TST_Status != DX_SUCCESS)
      {
        ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"Encrypted data is different than expected \n"));
        return DX_FAILURE;    
      }
    }
    
    /* Finish the operation */
    TST_Status = CRYS_AES_Finish(&TST_AesContext);
    
    TST_ATP_CHECK_ERROR_RETURN((DxError_t)TST_Status,
                               "CRYS_AES_Finish",
                               KeyData->TST_Name,
                               TST_MasterTestName);                                          
  }
  
  return DX_SUCCESS;

EXIT_ON_ERROR:
  {
    return DX_FAILURE;
  }
}

/*****************************************************************************
 * Function Name:                                                           
 *  KMNG_TST_Activate_And_Use_AES_Integrate      
 *  
 * Inputs:
 *  None
 *
 * Outputs: 
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *  This function activate a key from a given key ring and uses it.
 *  
 * Algorithm:       
 * 1. Activate the AES key
 * 2. AES Encrypt/Decrypt data block
 *
 ******************************************************************************/
DxError_t KMNG_TST_Activate_And_Use_AES_Integrate(TST_KeysSym_TestDataStruct* KeyData, 
                                                  DxUint8_t* KeyRing, DxUint8_t VerifyOutput)
{
  /********* TST Variables Definitions ************/
  DxError_t               TST_Status = DX_SUCCESS;
  DxChar                  TST_MasterTestName[TST_FUNC_STRING_LEN] = "KMNG_TST_Activate_And_Use_AES_NonIntegrate";  
  DxUint8_t               TST_Output[TST_KMNG_MAX_SIZE_DATA];
  DxUint8_t*              TST_Input_ptr;
  DxUint8_t               TST_index; 
  CRYS_AES_EncryptMode_t  TST_OpMode;
  
  ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"********************************************************************* \n"));
  ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"Activate and use %s \n",KeyData->TST_Name));      
  ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"********************************************************************* \n"));
  
  /********* Activate AES key  ************/  
  TST_Status = KMNG_ActivateAESKey(KeyData->TST_Pwd, KeyData->TST_PwdLen,
                                   KeyData->TST_KeyPwd, KeyData->TST_KeyPwdLen,
                                   KeyData->TST_KeyID, KeyRing,
                                   TST_AESWrappedKey);

  TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                             "KMNG_ActivateAESKey",
                             KeyData->TST_Name,
                             TST_MasterTestName);
                                             
    
  /********* AES Encrypt/Decrypt operation ************/
  for(TST_index = 0; TST_index < CRYS_AES_EncryptNumOfOptions; TST_index++)
  {
    if (TST_index == 0)
      TST_OpMode = CRYS_AES_Encrypt;
    else
      TST_OpMode = CRYS_AES_Decrypt;

    /* In AES MAC mode there is no decryption only encryption */
    if (KeyData->TST_AESMode == CRYS_AES_MAC_mode && TST_OpMode == CRYS_AES_Decrypt)
      break;

    /* In AES CTR mode there is only encrypt */
    if (KeyData->TST_AESMode == CRYS_AES_CTR_mode && TST_OpMode == CRYS_AES_Decrypt)
      TST_OpMode = CRYS_AES_Encrypt;

    /* copy input to output in case of encrypt*/
    if (TST_OpMode == CRYS_AES_Encrypt)
    {      
      DX_VOS_FastMemCpy(TST_Output, KeyData->TST_Data, KeyData->TST_DataLen);  
      TST_Input_ptr = &KeyData->TST_Data[0];
    }
    else
    {
      TST_Input_ptr = &KeyData->TST_OutputData[0];
    }
        
    TST_Status = CRYS_KMNG_AES(TST_AESWrappedKey,
            									KeyData->TST_IV,       
            									TST_OpMode, 
            									KeyData->TST_AESMode,       
            									TST_Input_ptr,        
            									KeyData->TST_DataLen,         
            									KeyData->TST_OutputData);   
  										              
    TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                               "CRYS_KMNG_AES",
                               KeyData->TST_Name,
                               TST_MasterTestName);
                                               
    
    /* Verify the output if needed */
    if (VerifyOutput == TST_KMNG_VERIFY_OUTPUT || TST_OpMode == CRYS_AES_Decrypt)
    {
      if (TST_OpMode == CRYS_AES_Encrypt)
      {
        if (KeyData->TST_AESMode == CRYS_AES_MAC_mode)
          TST_Status = (DxUint32_t)DX_VOS_MemCmp(KeyData->TST_ExpectedOutput,KeyData->TST_OutputData,TST_KMNG_MAC_SIZE_BYTE);
        else
          TST_Status = (DxUint32_t)DX_VOS_MemCmp(KeyData->TST_ExpectedOutput,KeyData->TST_OutputData,KeyData->TST_DataLen);
      }
      else
        TST_Status = (DxUint32_t)DX_VOS_MemCmp(KeyData->TST_Data,KeyData->TST_OutputData,KeyData->TST_DataLen);  
      
      if (TST_Status != DX_SUCCESS)
      {
        ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"Encrypted data is different than expected \n"));
        return DX_FAILURE;    
      }
    }
    
  }
  
  return DX_SUCCESS;
EXIT_ON_ERROR:
  {
    return DX_FAILURE;
  }
}

/*****************************************************************************
* Function Name:                                                           
*  KMNG_TST_Activate_And_Use_AES_WrapUnwrap      
*  
* Inputs:
*  None
*
* Outputs: 
*  CRYSError_t - Function error return                                     
*
* Description:                                                                 
*  This function activate a key from a given key ring and uses it.
*  
* Algorithm:       
* 1. Activate the AES key
* 2. AES wrap data block
* 3. AES Unwrap data block
*
******************************************************************************/
DxError_t KMNG_TST_Activate_And_Use_AES_WrapUnwrap(TST_KeysSym_TestDataStruct* KeyData, 
                                                   DxUint8_t* KeyRing, DxUint8_t VerifyOutput)
{
  /********* TST Variables Definitions ************/
  DxError_t               TST_Status = DX_SUCCESS;
  DxChar                  TST_MasterTestName[TST_FUNC_STRING_LEN] = "KMNG_TST_Activate_And_Use_AES_WrapUnwrap";  
  DxUint8_t               TST_Output[TST_KMNG_MAX_SIZE_DATA]; 
  DxUint32_t              TST_OutputLenTemp = KeyData->TST_OutputLen;

  ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"********************************************************************* \n"));
  ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"Activate and use %s \n",KeyData->TST_Name));      
  ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"********************************************************************* \n"));
  
  /********* Activate AES key  ************/  
  TST_Status = KMNG_ActivateAESKey(KeyData->TST_Pwd, KeyData->TST_PwdLen,
                                   KeyData->TST_KeyPwd, KeyData->TST_KeyPwdLen,
                                   KeyData->TST_KeyID, KeyRing,
                                   TST_AESWrappedKey);

  TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                            "KMNG_ActivateAESKey",
                            KeyData->TST_Name,
                            TST_MasterTestName);



  /********* AES Wrap/Unwrap operation ************/
  /* AES Wrap data */
  TST_Status = CRYS_KMNG_AESWrap(TST_AESWrappedKey,      
                                KeyData->TST_Data,        
                                KeyData->TST_DataLen,         
                                TST_Output,
                                &TST_OutputLenTemp);   

  TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                             "CRYS_KMNG_AESWrap",
                              KeyData->TST_Name,
                              TST_MasterTestName);

  /* Verify the output if needed */
  if (VerifyOutput == TST_KMNG_VERIFY_OUTPUT)
  {
    TST_Status = (DxUint32_t)DX_VOS_MemCmp(KeyData->TST_ExpectedOutput,TST_Output,TST_OutputLenTemp);
    if (TST_Status != DX_SUCCESS)
    {
      ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"Encrypted data is different than expected \n"));
      return DX_FAILURE;    
    }
  }

  /* AES Wrap data */
  TST_Status = CRYS_KMNG_AESUnwrap(TST_AESWrappedKey,      
                                  TST_Output,        
                                  KeyData->TST_OutputLen,         
                                  TST_Output,
                                  &TST_OutputLenTemp);   

  TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                             "CRYS_KMNG_AESUnwrap",
                             KeyData->TST_Name,
                             TST_MasterTestName);

  TST_Status = (DxUint32_t)DX_VOS_MemCmp(KeyData->TST_Data,TST_Output,TST_OutputLenTemp);
  if (TST_Status != DX_SUCCESS)
  {
    ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"Decrypted data is different than expected \n"));
    return DX_FAILURE;    
  }

  return DX_SUCCESS;
EXIT_ON_ERROR:
  {
    return DX_FAILURE;
  }
}

/*****************************************************************************
* Function Name:                                                           
*  KMNG_TST_Activate_And_Use_DES_NonIntegrate      
*  
* Inputs:
*  None
*
* Outputs: 
*  CRYSError_t - Function error return                                     
*
* Description:                                                                 
*  This function activate a key from a given key ring and uses it.
*  
* Algorithm:       
* 1. Activate the DES key
* 2. Initialize encryption operation
* 3. Encrypt data block
* 4. Compare the output to expected data (if required)
* 5. Finish the operation
* 6. Repeat steps 2-5 for decrypt operation  
*
******************************************************************************/
DxError_t KMNG_TST_Activate_And_Use_DES_NonIntegrate(TST_KeysSym_TestDataStruct* KeyData, 
                                                     DxUint8_t* KeyRing, DxUint8_t VerifyOutput)
{
  /********* TST Variables Definitions ************/
  DxError_t               TST_Status = DX_SUCCESS;
  CRYS_DESUserContext_t   TST_DesContext;  
  DxChar                  TST_MasterTestName[TST_FUNC_STRING_LEN] = "KMNG_TST_Activate_And_Use_DES_NonIntegrate";  
  DxUint8_t               TST_Output[TST_KMNG_MAX_SIZE_DATA];
  DxUint8_t*              TST_Input_ptr; 
  DxUint8_t               TST_index; 
  CRYS_DES_EncryptMode_t  TST_OpMode;

  ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"********************************************************************* \n"));
  ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"Activate and use %s \n",KeyData->TST_Name));      
  ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"********************************************************************* \n"));
  
  /********* Activate DES key  ************/  
  TST_Status = KMNG_ActivateDESKey(KeyData->TST_Pwd, KeyData->TST_PwdLen,
                                   KeyData->TST_KeyPwd, KeyData->TST_KeyPwdLen,
                                   KeyData->TST_KeyID, KeyRing,
                                   TST_DESWrappedKey );
 
  TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                             "KMNG_ActivateDESKey",
                             KeyData->TST_Name,
                             TST_MasterTestName);


  /********* Initialize the activated DES key ************/    
  /********* Encrypt\Decrypt operation ************/
  for(TST_index = 0; TST_index < CRYS_DES_EncryptNumOfOptions; TST_index++)
  {
    if (TST_index == 0)
      TST_OpMode = CRYS_DES_Encrypt;
    else
      TST_OpMode = CRYS_DES_Decrypt;

    /* Initialize DES context */  
    TST_Status = CRYS_KMNG_DES_Init(TST_DESWrappedKey,
                                   &TST_DesContext,
                                   KeyData->TST_IV,
                                   TST_OpMode,
                                   KeyData->TST_DESMode);

    TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                               "CRYS_KMNG_DES_Init",
                               KeyData->TST_Name,
                               TST_MasterTestName);

    /* copy input to output in case of encrypt*/
    if (TST_OpMode == CRYS_DES_Encrypt)
    {      
      DX_VOS_FastMemCpy(TST_Output, KeyData->TST_Data, KeyData->TST_DataLen);  
      TST_Input_ptr = KeyData->TST_Data;
    }
    else
    {
      TST_Input_ptr = KeyData->TST_OutputData;
    }

    /* Encrypt\Decrypt data */
    TST_Status = CRYS_DES_Block(&TST_DesContext,   
                               TST_Input_ptr,     
                               KeyData->TST_DataLen,
                               KeyData->TST_OutputData); 

    TST_ATP_CHECK_ERROR_RETURN((DxError_t)TST_Status,
                               "CRYS_DES_Block",
                               KeyData->TST_Name,
                               TST_MasterTestName);

    /* Verify the output if needed */
    if (VerifyOutput == TST_KMNG_VERIFY_OUTPUT || TST_OpMode == CRYS_DES_Decrypt)
    {
      if (TST_OpMode == CRYS_DES_Encrypt)
      { 
        TST_Status = (DxUint32_t)DX_VOS_MemCmp(KeyData->TST_ExpectedOutput,KeyData->TST_OutputData,KeyData->TST_DataLen);
      }
      else
        TST_Status = (DxUint32_t)DX_VOS_MemCmp(KeyData->TST_Data,KeyData->TST_OutputData,KeyData->TST_DataLen);  

      if (TST_Status != DX_SUCCESS)
      {
        ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"Encrypted data is different than expected \n"));
        return DX_FAILURE;    
      }
    }

    /* Finish the operation */
    TST_Status = CRYS_DES_Free(&TST_DesContext);

    TST_ATP_CHECK_ERROR_RETURN((DxError_t)TST_Status,
                               "CRYS_DES_Free",
                               KeyData->TST_Name,
                               TST_MasterTestName);
  }

    return DX_SUCCESS;
EXIT_ON_ERROR:
  {
    return DX_FAILURE;
  }
}

/*****************************************************************************
* Function Name:                                                           
*  KMNG_TST_Activate_And_Use_DES_Integrate      
*  
* Inputs:
*  None
*
* Outputs: 
*  CRYSError_t - Function error return                                     
*
* Description:                                                                 
*  This function activate a key from a given key ring and uses it.
*  
* Algorithm:       
* 1. Activate the DES key
* 2. Do encryption operation
* 3. Compare the output to expected data (if required)
* 4. Repeat steps 2-3 for decrypt operation  
*
******************************************************************************/
DxError_t KMNG_TST_Activate_And_Use_DES_Integrate(TST_KeysSym_TestDataStruct* KeyData, 
                                                  DxUint8_t* KeyRing, DxUint8_t VerifyOutput)
{
  /********* TST Variables Definitions ************/
  DxError_t               TST_Status = DX_SUCCESS;
  DxChar                  TST_MasterTestName[TST_FUNC_STRING_LEN] = "KMNG_TST_Activate_And_Use_DES_Integrate";  
  DxUint8_t               TST_Output[TST_KMNG_MAX_SIZE_DATA];
  DxUint8_t*              TST_Input_ptr; 
  DxUint8_t               TST_index;   
  CRYS_DES_EncryptMode_t  TST_OpMode;

  ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"********************************************************************* \n"));
  ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"Activate and use %s \n",KeyData->TST_Name));      
  ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"********************************************************************* \n"));
  
  /********* Activate DES key  ************/  
  TST_Status = KMNG_ActivateDESKey(KeyData->TST_Pwd, KeyData->TST_PwdLen,
                                   KeyData->TST_KeyPwd, KeyData->TST_KeyPwdLen,
                                   KeyData->TST_KeyID, KeyRing,
                                   TST_DESWrappedKey );

  TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                             "KMNG_ActivateDESKey",
                             KeyData->TST_Name,
                             TST_MasterTestName);

  /********* Encrypt\Decrypt operation ************/
  for(TST_index = 0; TST_index < CRYS_DES_EncryptNumOfOptions; TST_index++)
  {
    if (TST_index == 0)
      TST_OpMode = CRYS_DES_Encrypt;
    else
      TST_OpMode = CRYS_DES_Decrypt;

    /* copy input to output in case of encrypt*/
    if (TST_OpMode == CRYS_DES_Encrypt)
    {      
      DX_VOS_FastMemCpy(TST_Output, KeyData->TST_Data, KeyData->TST_DataLen);
      TST_Input_ptr = KeyData->TST_Data;
    }
    else
    {
      TST_Input_ptr = KeyData->TST_OutputData;
    }

    /* Encrypt\Decrypt data */
    
    TST_Status = CRYS_KMNG_DES(TST_DESWrappedKey,
				  			  KeyData->TST_IV,
				  			  TST_OpMode,
				  			  KeyData->TST_DESMode,             
				  			  TST_Input_ptr,              
				  			  KeyData->TST_DataLen,              
				  			  KeyData->TST_OutputData);
    
    TST_ATP_CHECK_ERROR_RETURN((DxError_t)TST_Status,
                               "CRYS_KMNG_DES",
                               KeyData->TST_Name,
                               TST_MasterTestName);

    /* Verify the output if needed */
    if (VerifyOutput == TST_KMNG_VERIFY_OUTPUT || TST_OpMode == CRYS_DES_Decrypt)
    {
      if (TST_OpMode == CRYS_DES_Encrypt)
      { 
        TST_Status = (DxUint32_t)DX_VOS_MemCmp(KeyData->TST_ExpectedOutput,KeyData->TST_OutputData,KeyData->TST_DataLen);
      }
      else
        TST_Status = (DxUint32_t)DX_VOS_MemCmp(KeyData->TST_Data,KeyData->TST_OutputData,KeyData->TST_DataLen);  

      if (TST_Status != DX_SUCCESS)
      {
        ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"Encrypted data is different than expected \n"));
        return DX_FAILURE;    
      }
    }
  }

      return DX_SUCCESS;
EXIT_ON_ERROR:
  {
    return DX_FAILURE;
  }
}

/*****************************************************************************
 * Function Name:                                                           
 *  KMNG_TST_Activate_And_Use_HMAC_NonIntegrate      
 *  
 * Inputs:
 *  None
 *
 * Outputs: 
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *  This function activate a key from a given key ring and uses it.
 *  
 * Algorithm:       
 * 1. Activate the HMAC key
 * 2. Do HMAC operation
 * 3. Verify the result (if required)  
 *
 ******************************************************************************/
DxError_t KMNG_TST_Activate_And_Use_HMAC_NonIntegrate(TST_KeysSym_TestDataStruct* KeyData, 
                                                     DxUint8_t* KeyRing, DxUint8_t VerifyOutput)
{
  /********* TST Variables Definitions ************/
  DxError_t               TST_Status = DX_SUCCESS;
  CRYS_HMACUserContext_t  TST_HMACContext;  
  DxChar                  TST_MasterTestName[TST_FUNC_STRING_LEN] = "KMNG_TST_Activate_And_Use_HMAC_NonIntegrate";  
  CRYS_HASH_Result_t      TST_Output; 
  
  ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"********************************************************************* \n"));
  ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"Activate and use %s \n",KeyData->TST_Name));      
  ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"********************************************************************* \n"));
  
  /********* Activate HMAC key  ************/  
  TST_Status = KMNG_ActivateHMACKey(KeyData->TST_Pwd, KeyData->TST_PwdLen,
                                    KeyData->TST_KeyPwd, KeyData->TST_KeyPwdLen,
                                    KeyData->TST_KeyID, KeyRing,
                                    TST_HMACWrappedKey);

  TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                             "KMNG_ActivateHMACKey",
                             KeyData->TST_Name,
                             TST_MasterTestName);

  
  /********* Initialize the activated HMAC key ************/   
  /* Initialize HMAC context */ 
  TST_Status = CRYS_KMNG_HMAC_Init(TST_HMACWrappedKey,
										               &TST_HMACContext, KeyData->TST_HASHMode);
										              
  TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                             "CRYS_KMNG_HMAC_Init",
                             KeyData->TST_Name,
                             TST_MasterTestName);
    
  /* Do HMAC operation*/
  TST_Status = CRYS_HMAC_Update(&TST_HMACContext,   
                               KeyData->TST_Data,     
                               KeyData->TST_DataLen); 
  
  TST_ATP_CHECK_ERROR_RETURN((DxError_t)TST_Status,
                             "CRYS_HMAC_Update",
                             KeyData->TST_Name,
                             TST_MasterTestName);
  
  /* Finish the operation */
  TST_Status = CRYS_HMAC_Finish(&TST_HMACContext ,
                               TST_Output);
  
  TST_ATP_CHECK_ERROR_RETURN((DxError_t)TST_Status,
                             "CRYS_HMAC_Finish",
                             KeyData->TST_Name,
                             TST_MasterTestName);                                            
    
  /* Verify the output if needed */
  if (VerifyOutput == TST_KMNG_VERIFY_OUTPUT)
  {      
    TST_Status = (DxUint32_t)DX_VOS_MemCmp(KeyData->TST_ExpectedOutput,TST_Output,KeyData->TST_OutputLen);            
    if (TST_Status != DX_SUCCESS)
    {
      ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"Output data is different than expected \n"));
      return DX_FAILURE;    
    }
  }  
  
        return DX_SUCCESS;
EXIT_ON_ERROR:
  {
    return DX_FAILURE;
  }
}

/*****************************************************************************
 * Function Name:                                                           
 *  KMNG_TST_Activate_And_Use_HMAC_Integrate      
 *  
 * Inputs:
 *  None
 *
 * Outputs: 
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *  This function activate a key from a given key ring and uses it.
 *  
 * Algorithm:       
 * 1. Activate the HMAC key
 * 2. Do HMAC operation
 * 3. Verify the result (if required)  
 *
 ******************************************************************************/
DxError_t KMNG_TST_Activate_And_Use_HMAC_Integrate(TST_KeysSym_TestDataStruct* KeyData, 
                                                   DxUint8_t* KeyRing, DxUint8_t VerifyOutput)
{
  /********* TST Variables Definitions ************/
  DxError_t               TST_Status = DX_SUCCESS;
  DxChar                  TST_MasterTestName[TST_FUNC_STRING_LEN] = "KMNG_TST_Activate_And_Use_HMAC_Integrate";  
  CRYS_HASH_Result_t      TST_Output; 
  
  ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"********************************************************************* \n"));
  ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"Activate and use %s \n",KeyData->TST_Name));      
  ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"********************************************************************* \n"));
  
  /********* Activate HMAC key  ************/  
  TST_Status = KMNG_ActivateHMACKey(KeyData->TST_Pwd, KeyData->TST_PwdLen,
                                    KeyData->TST_KeyPwd, KeyData->TST_KeyPwdLen,
                                    KeyData->TST_KeyID, KeyRing,
                                    TST_HMACWrappedKey);

  TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                             "KMNG_ActivateHMACKey",
                             KeyData->TST_Name,
                             TST_MasterTestName);
                                             

  
#ifdef LLF_HASH_SHA384_SHA512_NOT_SUPPORTED
  if(KeyData->TST_HASHMode == CRYS_HASH_SHA384_mode ||  KeyData->TST_HASHMode == CRYS_HASH_SHA512_mode)
  {
    goto end_function;
  }
#endif
  /********* Initialize the activated HMAC key ************/   
  /* Do HMAC operation */ 
  TST_Status = CRYS_KMNG_HMAC(TST_HMACWrappedKey,
										         KeyData->TST_HASHMode,
										         KeyData->TST_Data,     
                             KeyData->TST_DataLen,
                             TST_Output);
										              
  TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                             "CRYS_KMNG_HMAC",
                             KeyData->TST_Name,
                             TST_MasterTestName);
                                             

    
  /* Verify the output if needed */
  if (VerifyOutput == TST_KMNG_VERIFY_OUTPUT)
  {      
    TST_Status = (DxUint32_t)DX_VOS_MemCmp(KeyData->TST_ExpectedOutput,TST_Output,KeyData->TST_OutputLen);            
    if (TST_Status != DX_SUCCESS)
    {
      ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"Output data is different than expected \n"));
      return DX_FAILURE;    
    }
  }  
  
#ifdef LLF_HASH_SHA384_SHA512_NOT_SUPPORTED
end_function:
#endif
          return DX_SUCCESS;
EXIT_ON_ERROR:
  {
    return DX_FAILURE;
  }
}

/*****************************************************************************
 * Function Name:                                                           
 *  KMNG_TST_Activate_And_Use_RSA      
 *  
 * Inputs:
 *  None
 *
 * Outputs: 
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *  This function activate a key from a given key ring and uses it.
 *  
 * Algorithm:       
 * 1. Activate the RSA key for encryption
 * 2. Do RSA encrypt (PKCS1v1_5, PKCS1v2_1 or PRIMITIVES)
 * 3. Verify the result (if required)
 * 4. Activate the RSA key for decryption
 * 2. Do RSA decrypt (PKCS1v1_5, PKCS1v2_1 or PRIMITIVES)
 * 3. Verify if the result is equal to the input data
 *
 ******************************************************************************/
DxError_t KMNG_TST_Activate_And_Use_RSA_EncryptDecrypt(TST_KeysRSA_TestDataStruct* KeyData, 
                                        DxUint8_t* KeyRing, DxUint8_t VerifyOutput)
{
  /********* TST Variables Definitions ************/
  DxError_t                                     TST_Status = DX_SUCCESS;
  DxChar                                        TST_MasterTestName[TST_FUNC_STRING_LEN] = "KMNG_TST_Activate_And_Use_RSA";  
  CRYS_RSAPrimeData_t							              PrimeData;
  DxUint8_t                                     TST_Output[TST_KMNG_MAX_SIZE_DATA] = {0};
  DxUint16_t				                            TST_OutputSize = TST_KMNG_MAX_SIZE_DATA;
  DxUint16_t                                    TST_DataSize = KeyData->TST_DataLen;  
  
  ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"********************************************************************* \n"));
  ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"Activate and use %s \n",KeyData->TST_Name));      
  ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"********************************************************************* \n"));
  
  /* The data size that is encrypted is different for each scheme */
  if (KeyData->TST_PKCS1_ver == CRYS_PKCS1_VER15)
    TST_DataSize = KeyData->TST_DataLen - TST_KMNG_LESS_DATA_FOR_PKCS1V15_ENCRYPT;
  if (KeyData->TST_PKCS1_ver == CRYS_PKCS1_VER21)
  {
    switch (KeyData->TST_HASHMode)
    {
      case CRYS_RSA_HASH_SHA1_mode:
        TST_DataSize = KeyData->TST_DataLen - TST_KMNG_LESS_DATA_FOR_OAEP_ENCRYPT_SHA1;
        break;
      case CRYS_RSA_HASH_SHA224_mode:
        TST_DataSize = KeyData->TST_DataLen - TST_KMNG_LESS_DATA_FOR_OAEP_ENCRYPT_SHA224;
        break;
      case CRYS_RSA_HASH_SHA256_mode:
        TST_DataSize = KeyData->TST_DataLen - TST_KMNG_LESS_DATA_FOR_OAEP_ENCRYPT_SHA256;
        break;
      case CRYS_RSA_HASH_SHA384_mode:
        TST_DataSize = KeyData->TST_DataLen - TST_KMNG_LESS_DATA_FOR_OAEP_ENCRYPT_SHA384;
        break;
      case CRYS_RSA_HASH_SHA512_mode:
        TST_DataSize = KeyData->TST_DataLen - TST_KMNG_LESS_DATA_FOR_OAEP_ENCRYPT_SHA512;
        break;
      default:
        return KMNG_TEST_FAIL;
    }
  }
  
  /******************* Encrypt *********************/
  /********* Activate RSA key - Encrypt ************/    
  if (KeyData->TST_KeyType != KMNG_KeyTypeRSAPriv )
  {  
    TST_Status = KMNG_ActivateRSAKey(KeyData->TST_Pwd, KeyData->TST_PwdLen,
                                     KeyData->TST_KeyPwd, KeyData->TST_KeyPwdLen,
                                     KeyData->TST_KeyID, KMNG_RsaKeyOperationEncrypt,
                                     KeyRing,TST_RSAWrappedKey);

    TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                               "KMNG_ActivateRSAKey",
                               KeyData->TST_Name,
                               TST_MasterTestName);
                                               

    
        
    /********* RSA Encrypt (PKCS#1V1_5, PKCS#1V2_1 or PRIMITIVES) ************/   
    if (KeyData->TST_PKCS1_ver == CRYS_PKCS1_VER21)
    {
      #ifdef LLF_HASH_SHA384_SHA512_NOT_SUPPORTED
      if(KeyData->TST_HASHMode != CRYS_HASH_SHA384_mode ||  KeyData->TST_HASHMode != CRYS_HASH_SHA512_mode)
      {
      #endif
        TST_Status = CRYS_KMNG_RSA_OAEP_Encrypt(TST_RSAWrappedKey,
                                                &PrimeData,KeyData->TST_HASHMode, DX_NULL,
                                                0, CRYS_PKCS1_MGF1, KeyData->TST_Data,
                                                TST_DataSize,TST_Output);
        
        TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                                   "CRYS_KMNG_RSA_OAEP_Encrypt",
                                   KeyData->TST_Name,
                                   TST_MasterTestName);
                                                 

     
      #ifdef LLF_HASH_SHA384_SHA512_NOT_SUPPORTED
      }
      #endif
                                                 
    }
    else   
    {
      if (KeyData->TST_PKCS1_ver == CRYS_PKCS1_VER15)
      {     
        TST_Status = CRYS_KMNG_RSA_PKCS1v15_Encrypt(TST_RSAWrappedKey,
                                                    &PrimeData, KeyData->TST_Data,
                                                    TST_DataSize,TST_Output); 
                                                     
        TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                                   "CRYS_KMNG_RSA_PKCS1v15_Encrypt",
                                   KeyData->TST_Name,
                                   TST_MasterTestName);
                                                 
                                                
      }
      /* PRIM encrypt*/
      else
      {
        TST_Status = CRYS_KMNG_RSA_PRIM_Encrypt(TST_RSAWrappedKey,
  												                     &PrimeData,
                      												 KeyData->TST_Data,
                      												 TST_DataSize,
                      												 TST_Output);
                      												 
        TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                                   "CRYS_KMNG_RSA_PRIM_Encrypt",
                                   KeyData->TST_Name,
                                   TST_MasterTestName);
                 												 
      }
    }
    
    #ifdef LLF_HASH_SHA384_SHA512_NOT_SUPPORTED
    if(KeyData->TST_HASHMode != CRYS_HASH_SHA384_mode ||  KeyData->TST_HASHMode != CRYS_HASH_SHA512_mode)
    {
    #endif  
    /* Verify the output if needed */
      if (VerifyOutput == TST_KMNG_VERIFY_OUTPUT)
      {      
        TST_Status = (DxUint32_t)DX_VOS_MemCmp(KeyData->TST_ExpectedOutput,TST_Output,KeyData->TST_OutputLen);            
        if (TST_Status != DX_SUCCESS)
        {
          ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"Output data is different than expected \n"));
          return KMNG_TEST_FAIL;    
        }
      }
    #ifdef LLF_HASH_SHA384_SHA512_NOT_SUPPORTED
    }
    #endif
  }
  
  /******************* Decrypt *********************/
  /********* Activate RSA key - Decrypt ************/    
  if (KeyData->TST_KeyType != KMNG_KeyTypeRSAPub)
  {
    TST_Status = KMNG_ActivateRSAKey(KeyData->TST_Pwd, KeyData->TST_PwdLen,
                                     KeyData->TST_KeyPwd, KeyData->TST_KeyPwdLen,
                                     KeyData->TST_KeyID, KMNG_RsaKeyOperationDecrypt,
                                     KeyRing,TST_RSAWrappedKey);

    TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                               "KMNG_ActivateRSAKey",
                               KeyData->TST_Name,
                               TST_MasterTestName);
                                               
 
    
    /********* RSA Decrypt (PKCS#1V1_5 or PKCS#1V2_1) ************/   
    if (KeyData->TST_PKCS1_ver == CRYS_PKCS1_VER21)
    {
      #ifdef LLF_HASH_SHA384_SHA512_NOT_SUPPORTED
      if(KeyData->TST_HASHMode != CRYS_HASH_SHA384_mode ||  KeyData->TST_HASHMode != CRYS_HASH_SHA512_mode)
      {
      #endif
        TST_Status = CRYS_KMNG_RSA_OAEP_Decrypt(TST_RSAWrappedKey,
                                                &PrimeData, KeyData->TST_HASHMode, 
                                                DX_NULL, 0,
                                                CRYS_PKCS1_MGF1, TST_Output,
                                                KeyData->TST_DataLen,TST_Output, &TST_OutputSize);
        
        TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                                   "CRYS_KMNG_RSA_OAEP_Decrypt",
                                   KeyData->TST_Name,
                                   TST_MasterTestName);
                                                 

      
      #ifdef LLF_HASH_SHA384_SHA512_NOT_SUPPORTED
      }
      #endif                                             
    }
    else
    {
      if (KeyData->TST_PKCS1_ver == CRYS_PKCS1_VER15)
      {
		TST_Status = CRYS_KMNG_RSA_PKCS1v15_Decrypt(TST_RSAWrappedKey,
                                                    &PrimeData,TST_Output,KeyData->TST_DataLen,
                                                    TST_Output, &TST_OutputSize); 
		                                                     
        TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                                   "CRYS_KMNG_RSA_PKCS1v15_Decrypt",
                                   KeyData->TST_Name,
                                   TST_MasterTestName);
		                                                 

      }
      else
      {
        TST_Status = CRYS_KMNG_RSA_PRIM_Decrypt(TST_RSAWrappedKey,
  												                     &PrimeData,TST_Output,TST_DataSize,
  												                     TST_Output);
  												                     
        TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                                   "CRYS_KMNG_RSA_PRIM_Decrypt",
                                   KeyData->TST_Name,
                                   TST_MasterTestName);
                                                       
        TST_OutputSize = KeyData->TST_OutputLen;
      }
    }
    
    #ifdef LLF_HASH_SHA384_SHA512_NOT_SUPPORTED
    if(KeyData->TST_HASHMode != CRYS_HASH_SHA384_mode ||  KeyData->TST_HASHMode != CRYS_HASH_SHA512_mode)
    {
    #endif   
      /* Verify the output */  
      TST_Status = (DxUint32_t)DX_VOS_MemCmp(KeyData->TST_Data,TST_Output,TST_OutputSize);            
      if (TST_Status != DX_SUCCESS)
      {
        ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"Output data is different than expected \n"));
        return DX_FAILURE;    
      }
    #ifdef LLF_HASH_SHA384_SHA512_NOT_SUPPORTED
    }
    #endif
  }
  
     return DX_SUCCESS;
EXIT_ON_ERROR:
  {
    return DX_FAILURE;
  }
}

/*****************************************************************************
 * Function Name:                                                           
 *  KMNG_TST_Activate_And_Use_RSA      
 *  
 * Inputs:
 *  None
 *
 * Outputs: 
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *  This function activate a key from a given key ring and uses it.
 *  
 * Algorithm:       
 * 1. Activate the RSA key for sign
 * 2. 
 *
 ******************************************************************************/
DxError_t KMNG_TST_Activate_And_Use_RSA_SignVerify(TST_KeysRSA_TestDataStruct* KeyData, 
                                                   DxUint8_t* KeyRing,
                                                   TST_KeysRSAHASHData_TestDataStruct* KeysOpRSAHASHData)
{
  /********* TST Variables Definitions ************/
  DxError_t                                     TST_Status = DX_SUCCESS;
  DxChar                                        TST_MasterTestName[TST_FUNC_STRING_LEN] = "KMNG_TST_Activate_And_Use_RSA_SignVerify";  
  CRYS_RSAPrivUserContext_t                     TST_PrivContext;
  CRYS_RSAPubUserContext_t                      TST_PubContext;
  DxUint8_t                                     TST_Output[TST_KMNG_MAX_SIZE_DATA] = {0};
  DxUint16_t				                            TST_OutputSize = TST_KMNG_MAX_SIZE_DATA;
  
  ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"********************************************************************* \n"));
  ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"Activate and use %s \n",KeyData->TST_Name));      
  ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"********************************************************************* \n"));

  #ifdef LLF_HASH_SHA384_SHA512_NOT_SUPPORTED
  if(KeyData->TST_HASHMode == CRYS_HASH_SHA384_mode ||  KeyData->TST_HASHMode == CRYS_HASH_SHA512_mode)
  {
    return  TST_Status;
  }
  #endif
  
  /******************* Sign *********************/
  /********* Activate RSA key - Sign (priv key) ************/    
  if (KeyData->TST_KeyType != KMNG_KeyTypeRSAPub)
  {  
    TST_Status = KMNG_ActivateRSAKey(KeyData->TST_Pwd, KeyData->TST_PwdLen,
                                    KeyData->TST_KeyPwd, KeyData->TST_KeyPwdLen,
                                    KeyData->TST_KeyID, KMNG_RsaKeyOperationDecrypt,
                                    KeyRing,TST_RSAWrappedKey);

    TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                               "KMNG_ActivateRSAKey",
                               KeyData->TST_Name,
                               TST_MasterTestName);
                                               
            
    /********* RSA sign (PKCS#1V1_5, PKCS#1V2_1 or PRIMITIVES) ************/   
    /* PKCS#1V1_5 */
    if (KeyData->TST_PKCS1_ver == CRYS_PKCS1_VER15)
    {
      switch (KeyData->TST_HASHMode)
      {
        /*Choose the HASH function to be used in the signature*/        
        case CRYS_RSA_HASH_MD5_mode:
        case CRYS_RSA_HASH_SHA1_mode:
          TST_Status = CRYS_KMNG_RSA_PKCS1v15_Sign(TST_RSAWrappedKey,
                                                   &TST_PrivContext,KeyData->TST_HASHMode,
                                                   KeyData->TST_Data,KeyData->TST_DataLen,
                                                   TST_Output,&TST_OutputSize);
          TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                                     "CRYS_KMNG_RSA_PKCS1v15_Sign",
                                     KeyData->TST_Name,
                                     TST_MasterTestName);
                                               

          break;
        case CRYS_RSA_After_SHA1_mode:        
          TST_Status = CRYS_KMNG_RSA_PKCS1v15_SHA1_Sign(TST_RSAWrappedKey,
                                                        &TST_PrivContext,KeysOpRSAHASHData->TST_SHA1Data,                                                        
                                                        TST_Output,&TST_OutputSize);
          TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                                     "CRYS_KMNG_RSA_PKCS1v15_SHA1_Sign",
                                     KeyData->TST_Name,
                                     TST_MasterTestName);
                                               

          break;
        case CRYS_RSA_After_MD5_mode:        
          TST_Status = CRYS_KMNG_RSA_PKCS1v15_MD5_Sign(TST_RSAWrappedKey,
                                                        &TST_PrivContext,KeysOpRSAHASHData->TST_MD5Data,                                                        
                                                        TST_Output,&TST_OutputSize);
          TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                                     "CRYS_KMNG_RSA_PKCS1v15_MD5_Sign",
                                     KeyData->TST_Name,
                                     TST_MasterTestName);
                                               

          break;
        case CRYS_RSA_After_SHA224_mode:        
          TST_Status = CRYS_KMNG_RSA_PKCS1v15_SHA224_Sign(TST_RSAWrappedKey,
                                                          &TST_PrivContext,KeysOpRSAHASHData->TST_SHA224Data,                                                          
                                                          TST_Output,&TST_OutputSize);
          TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                                     "CRYS_KMNG_RSA_PKCS1v15_SHA224_Sign",
                                     KeyData->TST_Name,
                                     TST_MasterTestName);
                                               

          break;
        case CRYS_RSA_After_SHA256_mode:        
          TST_Status = CRYS_KMNG_RSA_PKCS1v15_SHA256_Sign(TST_RSAWrappedKey,
                                                          &TST_PrivContext,KeysOpRSAHASHData->TST_SHA256Data,                                                          
                                                          TST_Output,&TST_OutputSize);
          TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                                     "CRYS_KMNG_RSA_PKCS1v15_SHA256_Sign",
                                     KeyData->TST_Name,
                                     TST_MasterTestName);
                                               

          break;
        case CRYS_RSA_After_SHA384_mode:
          TST_Status = CRYS_KMNG_RSA_PKCS1v15_SHA384_Sign(TST_RSAWrappedKey,
                                                          &TST_PrivContext,KeysOpRSAHASHData->TST_SHA384Data,                                                          
                                                          TST_Output,&TST_OutputSize);
          TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                                     "CRYS_KMNG_RSA_PKCS1v15_SHA384_Sign",
                                     KeyData->TST_Name,
                                     TST_MasterTestName);
                                               

          break;
        case CRYS_RSA_After_SHA512_mode:
          TST_Status = CRYS_KMNG_RSA_PKCS1v15_SHA512_Sign(TST_RSAWrappedKey,
                                                          &TST_PrivContext,KeysOpRSAHASHData->TST_SHA512Data,                                                          
                                                          TST_Output,&TST_OutputSize);
          TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                                     "CRYS_KMNG_RSA_PKCS1v15_SHA512_Sign",
                                     KeyData->TST_Name,
                                     TST_MasterTestName);
                                               

          break;
        default:
          return KMNG_TEST_FAIL;
      }
    }
    /* PKCS#1V2_1 */
    if (KeyData->TST_PKCS1_ver == CRYS_PKCS1_VER21)
    {
      switch (KeyData->TST_HASHMode)
      {
        /*Choose the HASH function to be used in the signature*/        
        case CRYS_RSA_HASH_MD5_mode:
        case CRYS_RSA_HASH_SHA1_mode:          
          TST_Status = CRYS_KMNG_RSA_PSS_Sign(TST_RSAWrappedKey,
                                              &TST_PrivContext,KeyData->TST_HASHMode,
                                              CRYS_PKCS1_MGF1,0,
                                              KeyData->TST_Data,KeyData->TST_DataLen,
                                              TST_Output,&TST_OutputSize);
          TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                                                     "CRYS_KMNG_RSA_PSS_Sign",
                                                     KeyData->TST_Name,
                                                     TST_MasterTestName);
                                               

          break;
        case CRYS_RSA_After_SHA1_mode:         
          TST_Status = CRYS_KMNG_RSA_PSS_SHA1_Sign(TST_RSAWrappedKey,
                                                   &TST_PrivContext,
                                                   CRYS_PKCS1_MGF1,0,
                                                   KeysOpRSAHASHData->TST_SHA1Data,
                                                   TST_Output,&TST_OutputSize);
          TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                                                     "CRYS_KMNG_RSA_PSS_SHA1_Sign",
                                                     KeyData->TST_Name,
                                                     TST_MasterTestName);
                                               

          break;        
        case CRYS_RSA_After_SHA224_mode:        
          TST_Status = CRYS_KMNG_RSA_PSS_SHA224_Sign(TST_RSAWrappedKey,
                                                     &TST_PrivContext,
                                                     CRYS_PKCS1_MGF1,0,
                                                     KeysOpRSAHASHData->TST_SHA224Data,
                                                     TST_Output,&TST_OutputSize);
          TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                                                     "CRYS_KMNG_RSA_PSS_SHA224_Sign",
                                                     KeyData->TST_Name,
                                                     TST_MasterTestName);
                                               

          break;
        case CRYS_RSA_After_SHA256_mode:        
          TST_Status = CRYS_KMNG_RSA_PSS_SHA256_Sign(TST_RSAWrappedKey,
                                                     &TST_PrivContext,
                                                     CRYS_PKCS1_MGF1,0,
                                                     KeysOpRSAHASHData->TST_SHA256Data,
                                                     TST_Output,&TST_OutputSize);
          TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                                                     "CRYS_KMNG_RSA_PSS_SHA256_Sign",
                                                     KeyData->TST_Name,
                                                     TST_MasterTestName);
                                               

          break;
        case CRYS_RSA_After_SHA384_mode:
          TST_Status = CRYS_KMNG_RSA_PSS_SHA384_Sign(TST_RSAWrappedKey,
                                                     &TST_PrivContext,
                                                     CRYS_PKCS1_MGF1,0,
                                                     KeysOpRSAHASHData->TST_SHA384Data,
                                                     TST_Output,&TST_OutputSize);
          TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                                                     "CRYS_KMNG_RSA_PSS_SHA384_Sign",
                                                     KeyData->TST_Name,
                                                     TST_MasterTestName);
                                               

          break;
        case CRYS_RSA_After_SHA512_mode:
          TST_Status = CRYS_KMNG_RSA_PSS_SHA512_Sign(TST_RSAWrappedKey,
                                                     &TST_PrivContext,
                                                     CRYS_PKCS1_MGF1,0,
                                                     KeysOpRSAHASHData->TST_SHA512Data,
                                                     TST_Output,&TST_OutputSize);
          TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                                                     "CRYS_KMNG_RSA_PSS_SHA512_Sign",
                                                     KeyData->TST_Name,
                                                     TST_MasterTestName);
                                               

          break;
        default:
          return DX_FAILURE;

      }
    } 
  }
  /******************* Verify *********************/
  /********* Activate RSA key - verify (pub key) ************/    
  if (KeyData->TST_KeyType != KMNG_KeyTypeRSAPriv)
  {  
    TST_Status = KMNG_ActivateRSAKey(KeyData->TST_Pwd, KeyData->TST_PwdLen,
                                     KeyData->TST_KeyPwd, KeyData->TST_KeyPwdLen,
                                     KeyData->TST_KeyID, KMNG_RsaKeyOperationEncrypt,
                                     KeyRing,TST_RSAWrappedKey);

    TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                               "KMNG_ActivateRSAKey",
                               KeyData->TST_Name,
                               TST_MasterTestName);
                                               

            
    /********* RSA sign (PKCS#1V1_5, PKCS#1V2_1 or PRIMITIVES) ************/   
    /* PKCS#1V1_5 */
    if (KeyData->TST_PKCS1_ver == CRYS_PKCS1_VER15)
    {
      switch (KeyData->TST_HASHMode)
      {
        /*Choose the HASH function to be used in the verification*/        
        case CRYS_RSA_HASH_MD5_mode:
        case CRYS_RSA_HASH_SHA1_mode:        
          TST_Status = CRYS_KMNG_RSA_PKCS1v15_Verify(TST_RSAWrappedKey,
                                                     &TST_PubContext,KeyData->TST_HASHMode,                                                   
                                                     KeyData->TST_Data,KeyData->TST_DataLen,
                                                     TST_Output);
          TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                                     "CRYS_KMNG_RSA_PKCS1v15_Verify",
                                     KeyData->TST_Name,
                                     TST_MasterTestName);
                                               

          break;
        case CRYS_RSA_After_SHA1_mode:        
          TST_Status = CRYS_KMNG_RSA_PKCS1v15_SHA1_Verify(TST_RSAWrappedKey,
                                                          &TST_PubContext,KeysOpRSAHASHData->TST_SHA1Data,
                                                          TST_Output);
          TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                                     "CRYS_KMNG_RSA_PKCS1v15_SHA1_Verify",
                                     KeyData->TST_Name,
                                     TST_MasterTestName);
                                               

          break;
        case CRYS_RSA_After_MD5_mode:        
          TST_Status = CRYS_KMNG_RSA_PKCS1v15_MD5_Verify(TST_RSAWrappedKey,
                                                          &TST_PubContext,KeysOpRSAHASHData->TST_MD5Data,
                                                          TST_Output);
          TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                                     "CRYS_KMNG_RSA_PKCS1v15_MD5_Verify",
                                     KeyData->TST_Name,
                                     TST_MasterTestName);
                                               

          break;
        case CRYS_RSA_After_SHA224_mode:        
          TST_Status = CRYS_KMNG_RSA_PKCS1v15_SHA224_Verify(TST_RSAWrappedKey,
                                                            &TST_PubContext,KeysOpRSAHASHData->TST_SHA224Data,
                                                            TST_Output);
          TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                                     "CRYS_KMNG_RSA_PKCS1v15_SHA224_Verify",
                                     KeyData->TST_Name,
                                     TST_MasterTestName);
                                               

          break;
        case CRYS_RSA_After_SHA256_mode:        
          TST_Status = CRYS_KMNG_RSA_PKCS1v15_SHA256_Verify(TST_RSAWrappedKey,
                                                            &TST_PubContext,KeysOpRSAHASHData->TST_SHA256Data,
                                                            TST_Output);
          TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                                     "CRYS_KMNG_RSA_PKCS1v15_SHA256_Verify",
                                     KeyData->TST_Name,
                                     TST_MasterTestName);
                                               

          break;
        case CRYS_RSA_After_SHA384_mode:
          TST_Status = CRYS_KMNG_RSA_PKCS1v15_SHA384_Verify(TST_RSAWrappedKey,
                                                            &TST_PubContext,KeysOpRSAHASHData->TST_SHA384Data,
                                                            TST_Output);
          TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                                     "CRYS_KMNG_RSA_PKCS1v15_SHA384_Verify",
                                     KeyData->TST_Name,
                                     TST_MasterTestName);
                                               

          break;
        case CRYS_RSA_After_SHA512_mode:
          TST_Status = CRYS_KMNG_RSA_PKCS1v15_SHA512_Verify(TST_RSAWrappedKey,
                                                            &TST_PubContext,KeysOpRSAHASHData->TST_SHA512Data,
                                                            TST_Output);
          TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                                     "CRYS_KMNG_RSA_PKCS1v15_SHA512_Verify",
                                     KeyData->TST_Name,
                                     TST_MasterTestName);
                                               

          break;
        default:
          return DX_FAILURE;

      }
    }  
    if (KeyData->TST_PKCS1_ver == CRYS_PKCS1_VER21)
    {
      /*Choose the HASH function to be used in the verification*/        
      switch (KeyData->TST_HASHMode)
      {    
        case CRYS_RSA_HASH_MD5_mode:
        case CRYS_RSA_HASH_SHA1_mode:        
          TST_Status = CRYS_KMNG_RSA_PSS_Verify(TST_RSAWrappedKey,
                                                &TST_PubContext,KeyData->TST_HASHMode,
                                                CRYS_PKCS1_MGF1,0,                                                   
                                                KeyData->TST_Data,KeyData->TST_DataLen,
                                                TST_Output);
          TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                                     "CRYS_KMNG_RSA_PSS_Verify",
                                     KeyData->TST_Name,
                                     TST_MasterTestName);
                                               

          break;
        case CRYS_RSA_After_SHA1_mode:        
          TST_Status = CRYS_KMNG_RSA_PSS_SHA1_Verify(TST_RSAWrappedKey,
                                                     &TST_PubContext,
                                                     CRYS_PKCS1_MGF1,0,                                                   
                                                     KeysOpRSAHASHData->TST_SHA1Data,
                                                     TST_Output);
          TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                                     "CRYS_KMNG_RSA_PSS_SHA1_Verify",
                                     KeyData->TST_Name,
                                     TST_MasterTestName);
                                               

          break;        
        case CRYS_RSA_After_SHA224_mode:        
          TST_Status = CRYS_KMNG_RSA_PSS_SHA224_Verify(TST_RSAWrappedKey,
                                                       &TST_PubContext,
                                                       CRYS_PKCS1_MGF1,0,                                                   
                                                       KeysOpRSAHASHData->TST_SHA224Data,
                                                       TST_Output);
          TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                                     "CRYS_KMNG_RSA_PSS_SHA224_Verify",
                                     KeyData->TST_Name,
                                     TST_MasterTestName);
                                               

          break;
        case CRYS_RSA_After_SHA256_mode:        
          TST_Status = CRYS_KMNG_RSA_PSS_SHA256_Verify(TST_RSAWrappedKey,
                                                       &TST_PubContext,
                                                       CRYS_PKCS1_MGF1,0,                                                   
                                                       KeysOpRSAHASHData->TST_SHA256Data,
                                                       TST_Output);
          TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                                     "CRYS_KMNG_RSA_PSS_SHA256_Verify",
                                     KeyData->TST_Name,
                                     TST_MasterTestName);
                                               

          break;
        case CRYS_RSA_After_SHA384_mode:
          TST_Status = CRYS_KMNG_RSA_PSS_SHA384_Verify(TST_RSAWrappedKey,
                                                       &TST_PubContext,
                                                       CRYS_PKCS1_MGF1,0,                                                   
                                                       KeysOpRSAHASHData->TST_SHA384Data,
                                                       TST_Output);
          TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                                     "CRYS_KMNG_RSA_PSS_SHA384_Verify",
                                     KeyData->TST_Name,
                                     TST_MasterTestName);
                                               

          break;
        case CRYS_RSA_After_SHA512_mode:
          TST_Status = CRYS_KMNG_RSA_PSS_SHA512_Verify(TST_RSAWrappedKey,
                                                       &TST_PubContext,
                                                       CRYS_PKCS1_NO_MGF,0,                                                   
                                                       KeysOpRSAHASHData->TST_SHA512Data,
                                                       TST_Output);
          TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                                     "CRYS_KMNG_RSA_PSS_SHA512_Verify",
                                     KeyData->TST_Name,
                                     TST_MasterTestName);
                                               

          break;
        default:
          return DX_FAILURE;
      }
        
    }
  }
       return DX_SUCCESS;
EXIT_ON_ERROR:
  {
    return DX_FAILURE;
  }
}


/*****************************************************************************
 * Function Name:                                                           
 *  KMNG_TST_Activate_And_Use_RSA_SignVerify_NonIntegrated      
 *  
 * Inputs:
 *  None
 *
 * Outputs: 
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *  This function activate a key from a given key ring and uses it.
 *  
 * Algorithm:       
 * 1. Activate the RSA key for sign
 * 2. 
 *
 ******************************************************************************/
DxError_t KMNG_TST_Activate_And_Use_RSA_SignVerify_NonIntegrated(TST_KeysRSA_TestDataStruct* KeyData, 
                                                                 DxUint8_t* KeyRing)
{
  /********* TST Variables Definitions ************/
  DxError_t                                     TST_Status=DX_SUCCESS;  
  DxChar                                        TST_MasterTestName[TST_FUNC_STRING_LEN] = "KMNG_TST_Activate_And_Use_RSA_SignVerify_NonIntegrated";  
  CRYS_RSAPrivUserContext_t                     TST_PrivContext;
  CRYS_RSAPubUserContext_t                      TST_PubContext;
  DxUint8_t                                     TST_Output[TST_KMNG_MAX_SIZE_DATA] = {0};
  DxUint16_t				                            TST_OutputSize = TST_KMNG_MAX_SIZE_DATA;
  
  ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"********************************************************************* \n"));
  ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"%s \n",KeyData->TST_Name));      
  ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"********************************************************************* \n"));

  #ifdef LLF_HASH_SHA384_SHA512_NOT_SUPPORTED
  if(KeyData->TST_HASHMode == CRYS_HASH_SHA384_mode ||  KeyData->TST_HASHMode == CRYS_HASH_SHA512_mode)
  {
    return  TST_Status;
  }
  #endif
  
  /******************* Sign *********************/
  /********* Activate RSA key - Sign (priv key) ************/    
  if (KeyData->TST_KeyType != KMNG_KeyTypeRSAPub)
  {  
    TST_Status = KMNG_ActivateRSAKey(KeyData->TST_Pwd, KeyData->TST_PwdLen,
                                    KeyData->TST_KeyPwd, KeyData->TST_KeyPwdLen,
                                    KeyData->TST_KeyID, KMNG_RsaKeyOperationDecrypt,
                                    KeyRing,TST_RSAWrappedKey);

    TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                               "KMNG_ActivateRSAKey",
                               KeyData->TST_Name,
                               TST_MasterTestName);
                                               

            
    /********* RSA sign (PKCS#1V1_5, PKCS#1V2_1 or PRIMITIVES) ************/   
    /* PKCS#1V1_5 */
    if (KeyData->TST_PKCS1_ver == CRYS_PKCS1_VER15)
    {
      TST_Status = CRYS_KMNG_RSA_PKCS1v15_SignInit(TST_RSAWrappedKey,
                                                  &TST_PrivContext,
                                                  KeyData->TST_HASHMode);
      TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                                 "CRYS_KMNG_RSA_PKCS1v15_SignInit",
                                 KeyData->TST_Name,
                                 TST_MasterTestName);
                                               

      /* PKCS1V1_5 Sign Update*/
      TST_Status = CRYS_RSA_PKCS1v15_SignUpdate(&TST_PrivContext,KeyData->TST_Data,
                                               KeyData->TST_DataLen);
      TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                                 "CRYS_RSA_PKCS1v15_SignUpdate",
                                 KeyData->TST_Name,
                                 TST_MasterTestName);                                              

      
      /* PKCS1V1_5 Sign Finish*/
      TST_Status = CRYS_RSA_PKCS1v15_SignFinish(&TST_PrivContext,TST_Output,
                                               &TST_OutputSize);
      TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                                 "CRYS_RSA_PKCS1v15_SignFinish",
                                 KeyData->TST_Name,
                                 TST_MasterTestName);                                              
    
    
    }/* PKCS#1V2_1 */
    if (KeyData->TST_PKCS1_ver == CRYS_PKCS1_VER21)
    {
      /* PSS Sign Init*/
      TST_Status = CRYS_KMNG_RSA_PSS_SignInit(TST_RSAWrappedKey,
                                             &TST_PrivContext,
                                             KeyData->TST_HASHMode,
                                             CRYS_PKCS1_MGF1,0);
      TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                                 "CRYS_KMNG_RSA_PSS_SignInit",
                                 KeyData->TST_Name,
                                 TST_MasterTestName);                                               

      
      /* PSS Sign Update*/
      TST_Status = CRYS_RSA_PSS_SignUpdate(&TST_PrivContext,KeyData->TST_Data,
                                          KeyData->TST_DataLen);
      TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                                 "CRYS_RSA_PSS_SignUpdate",
                                 KeyData->TST_Name,
                                 TST_MasterTestName);                                              

      
      /* PSS Sign Finish*/
      TST_Status = CRYS_RSA_PSS_SignFinish(&TST_PrivContext,TST_Output,
                                          &TST_OutputSize);
      TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                                 "CRYS_RSA_PSS_SignFinish",
                                 KeyData->TST_Name,
                                 TST_MasterTestName);                                              
                                     
      
    } 
  }
  /******************* Verify *********************/
  /********* Activate RSA key - verify (pub key) ************/    
  if (KeyData->TST_KeyType != KMNG_KeyTypeRSAPriv)
  {  
    TST_Status = KMNG_ActivateRSAKey(KeyData->TST_Pwd, KeyData->TST_PwdLen,
                                     KeyData->TST_KeyPwd, KeyData->TST_KeyPwdLen,
                                     KeyData->TST_KeyID, KMNG_RsaKeyOperationEncrypt,
                                     KeyRing,TST_RSAWrappedKey);

    TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                               "KMNG_ActivateRSAKey",
                               KeyData->TST_Name,
                               TST_MasterTestName);
                                               

            
    /********* RSA verify (PKCS#1V1_5, PKCS#1V2_1 or PRIMITIVES) ************/   
    /* PKCS#1V1_5 */
    if (KeyData->TST_PKCS1_ver == CRYS_PKCS1_VER15)
    {
      TST_Status = CRYS_KMNG_RSA_PKCS1v15_VerifyInit(TST_RSAWrappedKey,
                                                    &TST_PubContext,
                                                    KeyData->TST_HASHMode);
      TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                                 "CRYS_KMNG_RSA_PKCS1v15_VerifyInit",
                                 KeyData->TST_Name,
                                 TST_MasterTestName);
                                               

      
      /* PKCS1V1_5 Verify Update*/
      TST_Status = CRYS_RSA_PKCS1v15_VerifyUpdate(&TST_PubContext,
                                                 KeyData->TST_Data,KeyData->TST_DataLen);
                                                 
      TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                                 "CRYS_RSA_PKCS1v15_VerifyUpdate",
                                 KeyData->TST_Name,
                                 TST_MasterTestName);
                                               

      
      /* PKCS1V1_5 Verify Finish*/
      TST_Status = CRYS_RSA_PKCS1v15_VerifyFinish(&TST_PubContext,TST_Output);
      TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                                 "CRYS_RSA_PKCS1v15_VerifyFinish",
                                 KeyData->TST_Name,
                                 TST_MasterTestName);
                                               
  
    }  
    if (KeyData->TST_PKCS1_ver == CRYS_PKCS1_VER21)
    {
      TST_Status = CRYS_KMNG_RSA_PSS_VerifyInit(TST_RSAWrappedKey,
                                               &TST_PubContext, KeyData->TST_HASHMode, 
                                               CRYS_PKCS1_MGF1,0);
      TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                                 "CRYS_KMNG_RSA_PSS_VerifyInit",
                                 KeyData->TST_Name,
                                 TST_MasterTestName);
                                               

      /* PSS Verify Update*/
      TST_Status = CRYS_RSA_PSS_VerifyUpdate(&TST_PubContext,
                                            KeyData->TST_Data,KeyData->TST_DataLen);
                                                 
      TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                                 "CRYS_RSA_PSS_VerifyUpdate",
                                 KeyData->TST_Name,
                                 TST_MasterTestName);
                                               

      
      /* PSS Verify Finish*/
      TST_Status = CRYS_RSA_PSS_VerifyFinish(&TST_PubContext,TST_Output);
      TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                                 "CRYS_RSA_PSS_VerifyFinish",
                                 KeyData->TST_Name,
                                 TST_MasterTestName);
                                               

                 
    }
  }
         return DX_SUCCESS;
EXIT_ON_ERROR:
  {
    return DX_FAILURE;
  }
}

/*****************************************************************************
* Function Name:                                                           
*  KMNG_TST_Activate_And_Use_DH_GetSecretKey      
*  
* Inputs:
*  None
*
* Outputs: 
*  CRYSError_t - Function error return                                     
*
* Description:                                                                 
*  This function activate a key from a given key ring and uses it.
*  
* Algorithm:       
* 1. Activate DH key
* 2. Get secret key using the activated key
*
******************************************************************************/
DxError_t KMNG_TST_Activate_And_Use_DH_GetSecretKey(TST_KeysDHData_TestDataStruct* KeyData, 
                                                    DxUint8_t* KeyRing, TST_KeyDHServerData_Struct* ServerData)
{
    /********* TST Variables Definitions ************/
    DxError_t               TST_Status = DX_SUCCESS;  
    DxChar                  TST_MasterTestName[TST_FUNC_STRING_LEN] = "KMNG_TST_Activate_And_Use_DH_GetSecretKey";          
    DxUint16_t              TST_secretKeySize = TST_KMNG_MAX_DH_PARAM_SIZE;

    /********* Activate DH key  ************/     
    TST_Status = KMNG_ActivateDHKey(KeyData->TST_Pwd, KeyData->TST_PwdLen,
                                    KeyData->TST_KeyPwd, KeyData->TST_KeyPwdLen,
                                    KeyData->TST_KeyID, KeyRing,
                                    TST_DHWrappedKey);

    TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                               "KMNG_ActivateDHKey",
                               KeyData->TST_Name,
                               TST_MasterTestName);


    /********* Get secret key using the activated key  ************/       
    TST_Status = CRYS_KMNG_DH_PKCS3_GetSecretKey(TST_DHWrappedKey,
                                                 ServerData->TST_ServerPubKey,
                                                 ServerData->TST_ServerPubKeySize,
                                                 &TST_DHUserPubKey,
                                                 &TST_DHPrimeData,
                                                 KeyData->TST_SecretKey,
                                                 &TST_secretKeySize); 
    
    TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                             "CRYS_KMNG_DH_PKCS3_GetSecretKey",
                             KeyData->TST_Name,
                             TST_MasterTestName);   


   return DX_SUCCESS;

EXIT_ON_ERROR:
  {
    return DX_FAILURE;
  }
}
